<?php
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Queue management and WP-Cron processing.
 */
class Fifth_Social_Bot_Queue {

    public function __construct() {
        add_action( 'fifth_social_bot_process_queue_event', array( __CLASS__, 'process_queue' ) );
    }

    /**
     * Get queue (list of post IDs).
     *
     * @return array
     */
    public static function get_queue() {
        $queue = get_option( 'fifth_social_bot_queue', array() );
        if ( ! is_array( $queue ) ) {
            $queue = array();
        }
        $queue = array_filter( array_map( 'intval', $queue ) );
        return array_values( array_unique( $queue ) );
    }

    /**
     * Save queue.
     *
     * @param array $queue Queue.
     */
    public static function set_queue( $queue ) {
        if ( ! is_array( $queue ) ) {
            $queue = array();
        }
        $queue = array_filter( array_map( 'intval', $queue ) );
        $queue = array_values( array_unique( $queue ) );
        update_option( 'fifth_social_bot_queue', $queue, false );
    }

    /**
     * Enqueue posts.
     *
     * @param array $post_ids Post IDs.
     * @return array { added: int, skipped: int }
     */
    public static function enqueue_posts( $post_ids ) {
        $queue = self::get_queue();

        $added   = 0;
        $skipped = 0;

        foreach ( $post_ids as $post_id ) {
            $post_id = (int) $post_id;
            if ( $post_id <= 0 ) {
                continue;
            }

            if ( '1' === get_post_meta( $post_id, '_fifth_social_bot_posted', true ) ) {
                $skipped++;
                continue;
            }

            if ( in_array( $post_id, $queue, true ) ) {
                $skipped++;
                continue;
            }

            $queue[] = $post_id;
            update_post_meta( $post_id, '_fifth_social_bot_queued', '1' );
            $added++;
        }

        self::set_queue( $queue );

        if ( $added > 0 ) {
            self::schedule_queue();
        }

        return array(
            'added'   => $added,
            'skipped' => $skipped,
        );
    }

    /**
     * Queue latest posts/pages which are not yet posted or queued.
     *
     * @param int $count Number of posts to queue.
     * @return array { added: int, skipped: int, error: string }
     */
    public static function queue_latest( $count ) {
        $count = (int) $count;
        if ( $count <= 0 ) {
            return array(
                'added'   => 0,
                'skipped' => 0,
                'error'   => __( 'Invalid number.', '5th-social-bot' ),
            );
        }

        $settings = Fifth_Social_Bot_Core::get_settings();
        if ( empty( $settings['access_token'] ) ) {
            return array(
                'added'   => 0,
                'skipped' => 0,
                'error'   => __( 'Missing Mastodon access token.', '5th-social-bot' ),
            );
        }

        $queue = self::get_queue();

        $args = array(
            'post_type'      => array( 'post', 'page' ),
            'post_status'    => 'publish',
            'posts_per_page' => $count * 3,
            'orderby'        => 'date',
            'order'          => 'DESC',
            'no_found_rows'  => true,
        );

        $query = new WP_Query( $args );

        $ids = array();

        if ( $query->have_posts() ) {
            foreach ( $query->posts as $post ) {
                $post_id = $post->ID;

                if ( in_array( $post_id, $queue, true ) ) {
                    continue;
                }

                if ( '1' === get_post_meta( $post_id, '_fifth_social_bot_posted', true ) ) {
                    continue;
                }

                $ids[] = $post_id;

                if ( count( $ids ) >= $count ) {
                    break;
                }
            }
        }

        wp_reset_postdata();

        if ( empty( $ids ) ) {
            return array(
                'added'   => 0,
                'skipped' => 0,
                'error'   => __( 'No eligible posts found.', '5th-social-bot' ),
            );
        }

        $result = self::enqueue_posts( $ids );

        Fifth_Social_Bot_Logger::log(
            'Queued ' . $result['added'] . ' posts (bulk ' . $count . '). Skipped: ' . $result['skipped'],
            'info'
        );

        $result['error'] = '';

        return $result;
    }

    /**
     * Schedule queue processing event.
     */
    public static function schedule_queue() {
        if ( ! wp_next_scheduled( 'fifth_social_bot_process_queue_event' ) ) {
            wp_schedule_single_event( time() + 10, 'fifth_social_bot_process_queue_event' );
        }
    }

    /**
     * Process queue: send 1 post, re-schedule if there are more (every 5 minutes).
     */
    public static function process_queue() {
        $queue = self::get_queue();
        if ( empty( $queue ) ) {
            return;
        }

        $batch_size = 1; // Process 1 post at a time, every 5 minutes.
        $processed  = 0;
        $remaining  = array();

        foreach ( $queue as $post_id ) {
            if ( $processed >= $batch_size ) {
                $remaining[] = $post_id;
                continue;
            }

            if ( '1' === get_post_meta( $post_id, '_fifth_social_bot_posted', true ) ) {
                delete_post_meta( $post_id, '_fifth_social_bot_queued' );
                continue;
            }

            $ok = Fifth_Social_Bot_Core::send_status( $post_id );
            if ( ! $ok ) {
                $remaining[] = $post_id;
            }

            $processed++;
        }

        self::set_queue( $remaining );

        if ( ! empty( $remaining ) ) {
            // Schedule next batch in 5 minutes (300 seconds).
            wp_schedule_single_event( time() + 300, 'fifth_social_bot_process_queue_event' );
        } else {
            Fifth_Social_Bot_Logger::log( 'Queue empty, not rescheduling.', 'info' );
        }
    }
}
