<?php
/*
Plugin Name: 5th Social Bot
Description: Automatically posts to social.5th.ro (Mastodon) when you publish posts or pages in WordPress (excerpt + link + tags as hashtags), with a customizable template and queue/drip feed.
Version: 1.7.1
Author: SC ARC MEDIASOFT SRL
Author URI: https://e-agentie.ro
Text Domain: 5th-social-bot
Domain Path: /languages
License: GPL2
*/

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

define( 'FIFTH_SOCIAL_BOT_VERSION', '1.7.1' );
define( 'FIFTH_SOCIAL_BOT_INSTANCE', 'https://social.5th.ro' );
define( 'FIFTH_SOCIAL_BOT_PLUGIN_URL', 'https://plugin.5th.ro' );
define( 'FIFTH_SOCIAL_BOT_PLUGIN_FILE', __FILE__ );
define( 'FIFTH_SOCIAL_BOT_PLUGIN_DIR', plugin_dir_path( __FILE__ ) );

require_once FIFTH_SOCIAL_BOT_PLUGIN_DIR . 'includes/class-fifth-social-bot-logger.php';
require_once FIFTH_SOCIAL_BOT_PLUGIN_DIR . 'includes/class-fifth-social-bot-core.php';
require_once FIFTH_SOCIAL_BOT_PLUGIN_DIR . 'includes/class-fifth-social-bot-queue.php';
require_once FIFTH_SOCIAL_BOT_PLUGIN_DIR . 'includes/class-fifth-social-bot-boost.php';
require_once FIFTH_SOCIAL_BOT_PLUGIN_DIR . 'includes/class-fifth-social-bot-analytics.php';
require_once FIFTH_SOCIAL_BOT_PLUGIN_DIR . 'includes/class-fifth-social-bot-admin.php';
require_once FIFTH_SOCIAL_BOT_PLUGIN_DIR . 'includes/class-fifth-social-bot-shortcodes.php';
require_once FIFTH_SOCIAL_BOT_PLUGIN_DIR . 'includes/class-fifth-social-bot-support.php';
require_once FIFTH_SOCIAL_BOT_PLUGIN_DIR . 'includes/class-fifth-social-bot-terms.php';

/**
 * Initialize plugin.
 */
function fifth_social_bot_init() {
    // Core (posting logic).
    new Fifth_Social_Bot_Core();
    // Queue (WP-Cron processing).
    new Fifth_Social_Bot_Queue();
    // Boost (auto-boost functionality).
    new Fifth_Social_Bot_Boost();
    // Shortcodes and verification.
    new Fifth_Social_Bot_Shortcodes();
    // Admin UI.
    new Fifth_Social_Bot_Admin();
}
add_action( 'plugins_loaded', 'fifth_social_bot_init' );

/**
 * Extra meta links under plugin row.
 */
function fifth_social_bot_row_meta( $links, $file ) {
    if ( $file === plugin_basename( FIFTH_SOCIAL_BOT_PLUGIN_FILE ) ) {
        $support_url = 'https://plugin.5th.ro/support';
        
        $links[] = '<a href="' . esc_url( $support_url ) . '" target="_blank" rel="noopener noreferrer">'
            . esc_html__( 'Support', '5th-social-bot' ) . '</a>';
    }
    return $links;
}
add_filter( 'plugin_row_meta', 'fifth_social_bot_row_meta', 10, 2 );

/**
 * Extra action links in plugin list (Settings + support link).
 */
function fifth_social_bot_action_links( $links ) {
    $settings_url  = admin_url( 'admin.php?page=fifth-social-bot' );
    $settings_link = '<a href="' . esc_url( $settings_url ) . '">' . esc_html__( 'Settings', '5th-social-bot' ) . '</a>';
    $support_link  = '<a href="https://plugin.5th.ro" target="_blank" rel="noopener noreferrer">' . esc_html__( 'View details', '5th-social-bot' ) . '</a>';

    array_unshift( $links, $settings_link );
    $links[] = $support_link;

    return $links;
}
add_filter( 'plugin_action_links_' . plugin_basename( __FILE__ ), 'fifth_social_bot_action_links' );

/**
 * Check for plugin updates and show notification.
 */
function fifth_social_bot_check_version() {
    // Allow forcing check via ?force_version_check=1 (for testing).
    // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- GET parameter for testing only, no data modification.
    $force_check = isset( $_GET['force_version_check'] ) && '1' === sanitize_text_field( wp_unslash( $_GET['force_version_check'] ) ) && current_user_can( 'manage_options' );
    
    $last_check = get_transient( 'fifth_social_bot_version_check' );
    if ( false === $last_check || $force_check ) {
        $plugin_url = FIFTH_SOCIAL_BOT_PLUGIN_URL;
        $check_url = trailingslashit( $plugin_url ) . 'api/version-check.php?version=' . urlencode( FIFTH_SOCIAL_BOT_VERSION ) . '&domain=' . urlencode( home_url() );
        
        $response = wp_remote_get(
            $check_url,
            array(
                'timeout' => 10,
                'sslverify' => true,
            )
        );
        
        if ( ! is_wp_error( $response ) ) {
            $code = wp_remote_retrieve_response_code( $response );
            if ( 200 === $code ) {
                $body = wp_remote_retrieve_body( $response );
                $data = json_decode( $body, true );
                if ( is_array( $data ) && ! empty( $data['latest_version'] ) ) {
                    $latest_version = sanitize_text_field( $data['latest_version'] );
                    if ( version_compare( FIFTH_SOCIAL_BOT_VERSION, $latest_version, '<' ) ) {
                        update_option( 'fifth_social_bot_new_version', $latest_version );
                    } else {
                        delete_option( 'fifth_social_bot_new_version' );
                    }
                }
            }
        }
        
        // Check every 24 hours.
        set_transient( 'fifth_social_bot_version_check', time(), DAY_IN_SECONDS );
    }
}
add_action( 'admin_init', 'fifth_social_bot_check_version' );

/**
 * Show version update notification.
 */
function fifth_social_bot_version_notice() {
    if ( ! current_user_can( 'manage_options' ) ) {
        return;
    }
    
    $new_version = get_option( 'fifth_social_bot_new_version' );
    if ( ! empty( $new_version ) ) {
        $plugin_url = FIFTH_SOCIAL_BOT_PLUGIN_URL;
        $site_domain = wp_parse_url( home_url(), PHP_URL_HOST );
        $download_url = add_query_arg(
            array(
                'utm_source' => urlencode( $site_domain ),
                'utm_medium' => 'plugin',
                'utm_campaign' => 'update_notification',
            ),
            trailingslashit( $plugin_url ) . 'download'
        );
        
        ?>
        <div class="notice notice-info is-dismissible">
            <p>
                <strong><?php echo esc_html__( '5th Social Bot:', '5th-social-bot' ); ?></strong>
                <?php
                printf(
                    /* translators: %1$s: new version number, %2$s: download URL */
                    esc_html__( 'New version %1$s is available. Download from %2$s', '5th-social-bot' ),
                    esc_html( $new_version ),
                    '<a href="' . esc_url( $download_url ) . '" target="_blank" rel="noopener noreferrer">plugin.5th.ro</a>'
                );
                ?>
            </p>
        </div>
        <?php
    }
}
add_action( 'admin_notices', 'fifth_social_bot_version_notice' );

/**
 * Track plugin installation.
 */
function fifth_social_bot_track_installation() {
    $tracked = get_option( 'fifth_social_bot_installation_tracked' );
    if ( empty( $tracked ) ) {
        $plugin_url = FIFTH_SOCIAL_BOT_PLUGIN_URL;
        $site_domain = wp_parse_url( home_url(), PHP_URL_HOST );
        $track_url = trailingslashit( $plugin_url ) . 'api/track-install.php';
        
        wp_remote_post(
            $track_url,
            array(
                'timeout' => 5,
                'sslverify' => true,
                'body' => array(
                    'domain' => sanitize_text_field( $site_domain ),
                    'version' => FIFTH_SOCIAL_BOT_VERSION,
                    'wp_version' => get_bloginfo( 'version' ),
                ),
            )
        );
        
        update_option( 'fifth_social_bot_installation_tracked', time() );
    }
}
add_action( 'admin_init', 'fifth_social_bot_track_installation' );
