<?php
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Shortcodes and profile verification.
 */
class Fifth_Social_Bot_Shortcodes {

    public function __construct() {
        add_shortcode( 'mastodon_button', array( $this, 'mastodon_button_shortcode' ) );
        add_shortcode( 'mastodon_feed', array( $this, 'mastodon_feed_shortcode' ) );
        add_action( 'wp_head', array( $this, 'add_profile_verification' ) );
        add_action( 'wp_footer', array( $this, 'add_profile_verification_footer' ) );
    }

    /**
     * Get Mastodon username from settings.
     *
     * @return string
     */
    public static function get_mastodon_username() {
        $settings = Fifth_Social_Bot_Core::get_settings();
        return isset( $settings['mastodon_username'] ) ? trim( $settings['mastodon_username'] ) : '';
    }

    /**
     * Get Mastodon profile URL.
     *
     * @return string
     */
    public static function get_mastodon_profile_url() {
        $username = self::get_mastodon_username();
        if ( empty( $username ) ) {
            return '';
        }
        // Remove @ if present.
        $username = ltrim( $username, '@' );
        $instance_url = Fifth_Social_Bot_Core::get_instance_url();
        return trailingslashit( $instance_url ) . '@' . $username;
    }

    /**
     * Check if profile verification is enabled.
     *
     * @return bool
     */
    public static function is_verification_enabled() {
        $settings = Fifth_Social_Bot_Core::get_settings();
        return ! empty( $settings['enable_verification'] );
    }

    /**
     * Add profile verification meta tag in header.
     */
    public function add_profile_verification() {
        if ( ! self::is_verification_enabled() ) {
            return;
        }

        $username = self::get_mastodon_username();
        if ( empty( $username ) ) {
            return;
        }

        // Remove @ if present.
        $username = ltrim( $username, '@' );
        $instance_url = Fifth_Social_Bot_Core::get_instance_url();
        $instance_domain = str_replace( array( 'https://', 'http://' ), '', $instance_url );
        $full_username = $username . '@' . $instance_domain;

        echo '<meta name="fediverse:creator" content="@' . esc_attr( $full_username ) . '">' . "\n";
    }

    /**
     * Add profile verification link in footer.
     */
    public function add_profile_verification_footer() {
        if ( ! self::is_verification_enabled() ) {
            return;
        }

        $profile_url = self::get_mastodon_profile_url();
        if ( empty( $profile_url ) ) {
            return;
        }

        $username = self::get_mastodon_username();
        $username = ltrim( $username, '@' );

        echo '<a rel="me" href="' . esc_url( $profile_url ) . '" style="display:none;">Mastodon</a>' . "\n";
    }

    /**
     * Mastodon button shortcode.
     *
     * @param array  $atts    Shortcode attributes.
     * @param string $content Shortcode content.
     * @return string
     */
    /**
     * Check if debug mode is enabled.
     *
     * @return bool
     */
    private function is_debug_mode() {
        $settings = Fifth_Social_Bot_Core::get_settings();
        return ! empty( $settings['debug_mode'] );
    }
    
    /**
     * Log debug message if debug mode is enabled.
     *
     * @param string $message Message to log.
     * @param string $level   Log level (info|error).
     */
    private function debug_log( $message, $level = 'info' ) {
        if ( $this->is_debug_mode() ) {
            Fifth_Social_Bot_Logger::log( '[Shortcode] ' . $message, $level );
        }
    }

    public function mastodon_button_shortcode( $atts, $content = null ) {
        $this->debug_log( 'mastodon_button shortcode called with atts: ' . wp_json_encode( $atts ) );
        
        $atts = shortcode_atts(
            array(
                'url'      => '',
                'text'     => __( 'Mastodon', '5th-social-bot' ),
                'class'    => '',
                'style'    => 'button', // button, link, icon, modern, minimal, rounded, outline, gradient, pill, white, black, white-outline, black-outline, white-minimal, black-minimal
                'custom_css' => '', // Custom CSS class for additional styling.
                'size'       => 'medium', // small, medium, large
                'icon'       => 'yes', // yes, no
                'icon_color' => 'default', // default (blue), black, gray, red, white, green, purple
                'target'     => '_blank',
                'rel'        => 'me noopener noreferrer',
            ),
            $atts,
            'mastodon_button'
        );

        // Use profile URL from settings if URL not provided.
        if ( empty( $atts['url'] ) ) {
            $atts['url'] = self::get_mastodon_profile_url();
        }

        if ( empty( $atts['url'] ) ) {
            $this->debug_log( 'mastodon_button: Empty URL, returning empty string', 'error' );
            return '';
        }
        
        $url    = esc_url( $atts['url'] );
        $text   = esc_html( $atts['text'] );
        $class  = esc_attr( $atts['class'] );
        $target = esc_attr( $atts['target'] );
        $rel    = esc_attr( $atts['rel'] );
        $style  = esc_attr( $atts['style'] );
        $size   = esc_attr( $atts['size'] );
        
        $this->debug_log( 'mastodon_button: Rendering button with URL: ' . $url . ', style: ' . $style );

        // Validate size.
        if ( ! in_array( $size, array( 'small', 'medium', 'large' ), true ) ) {
            $size = 'medium';
        }

        // Build CSS classes.
        $css_classes = array( 'fsb-mastodon-button' );
        if ( ! empty( $class ) ) {
            $css_classes[] = $class;
        }
        
        // Style classes.
        $style_class = 'fsb-mastodon-' . $style . '-style';
        $valid_styles = array( 'button', 'link', 'icon', 'modern', 'minimal', 'rounded', 'outline', 'gradient', 'pill', 'white', 'black', 'white-outline', 'black-outline', 'white-minimal', 'black-minimal' );
        if ( in_array( $style, $valid_styles, true ) ) {
            $css_classes[] = $style_class;
        } else {
            $css_classes[] = 'fsb-mastodon-button-style';
        }
        
        // Add custom CSS class if provided (from 'class' or 'custom_css' attribute).
        $custom_class_attr = ! empty( $atts['custom_css'] ) ? $atts['custom_css'] : ( ! empty( $atts['class'] ) ? $atts['class'] : '' );
        if ( ! empty( $custom_class_attr ) ) {
            $custom_class = sanitize_html_class( $custom_class_attr );
            if ( ! empty( $custom_class ) ) {
                $css_classes[] = $custom_class;
            }
        }
        
        // Icon color class for icon style.
        $icon_color = sanitize_key( $atts['icon_color'] );
        $valid_icon_colors = array( 'default', 'black', 'gray', 'red', 'white', 'green', 'purple' );
        if ( ! in_array( $icon_color, $valid_icon_colors, true ) ) {
            $icon_color = 'default';
        }
        if ( 'icon' === $style ) {
            $css_classes[] = 'fsb-mastodon-icon-color-' . $icon_color;
        }
        
        // Size class.
        $css_classes[] = 'fsb-mastodon-size-' . $size;

        $css_class_string = implode( ' ', $css_classes );

        // Mastodon icon SVG - size varies by button size.
        $icon_size = 20;
        if ( 'small' === $size ) {
            $icon_size = 16;
        } elseif ( 'large' === $size ) {
            $icon_size = 24;
        }
        
        // Icon margin based on size.
        $icon_margin = '6px';
        if ( 'small' === $size ) {
            $icon_margin = '4px';
        } elseif ( 'large' === $size ) {
            $icon_margin = '8px';
        }
        
        $icon_svg = '';
        if ( 'yes' === $atts['icon'] || 'icon' === $style ) {
            $icon_svg = '<svg width="' . $icon_size . '" height="' . $icon_size . '" viewBox="0 0 24 24" fill="currentColor" aria-hidden="true" class="fsb-mastodon-icon" style="vertical-align:middle; display:inline-block;"><path d="M23.268 5.313c-.35-3.116-3.166-5.313-6.45-5.313-1.766 0-3.378.644-4.537 1.558a.95.95 0 0 0-.01.01c-1.16-.914-2.77-1.558-4.537-1.558-3.283 0-6.1 2.197-6.45 5.313C.532 6.075 0 7.833 0 9.872v6.278c0 1.655 1.343 3 3 3h1.11c.812 0 1.555-.5 1.832-1.263l1.222-3.409c.089-.25.32-.428.588-.428s.499.178.588.428l1.21 3.409c.278.763 1.02 1.263 1.833 1.263h2.11c.812 0 1.555-.5 1.832-1.263l1.222-3.409c.089-.25.32-.428.588-.428s.499.178.588.428l1.21 3.409c.278.763 1.02 1.263 1.833 1.263H21c1.655 0 3-1.345 3-3V9.872c0-2.04-.532-3.797-1.732-4.56zm-1.732 10.836c0 .552-.448 1-1 1h-1.11c-.312 0-.6-.192-.708-.487l-1.222-3.409c-.267-.75-1.037-1.26-1.832-1.26s-1.565.51-1.832 1.26l-1.21 3.409c-.108.295-.396.487-.708.487H8.11c-.312 0-.6-.192-.708-.487L6.18 13.993c-.267-.75-1.037-1.26-1.832-1.26s-1.565.51-1.832 1.26l-1.222 3.409c-.108.295-.396.487-.708.487H3c-.552 0-1-.448-1-1V9.872c0-1.66.432-3.035 1.134-3.643.45-.39 1.069-.62 1.732-.62 1.52 0 2.93.81 3.7 2.1.6 1 .6 2.3 0 3.3-.77 1.29-2.18 2.1-3.7 2.1-.276 0-.5.224-.5.5s.224.5.5.5c1.93 0 3.68-1.03 4.6-2.7.9-1.63.9-3.56 0-5.19-.92-1.67-2.67-2.7-4.6-2.7-1.096 0-2.12.3-2.984.826C2.32 4.227 2 5.562 2 7.22v8.93c0 .552.448 1 1 1h1.11c.312 0 .6-.192.708-.487l1.222-3.409c.267-.75 1.037-1.26 1.832-1.26s1.565.51 1.832 1.26l1.21 3.409c.108.295.396.487.708.487h2.11c.312 0 .6-.192.708-.487l1.222-3.409c.267-.75 1.037-1.26 1.832-1.26s1.565.51 1.832 1.26l1.21 3.409c.108.295.396.487.708.487H21c.552 0 1 .448 1 1v-8.93c0-1.658-.32-2.993-.984-4.074C19.152 3.642 18.128 3.342 17.032 3.342c-1.93 0-3.68 1.03-4.6 2.7-.9 1.63-.9 3.56 0 5.19.92 1.67 2.67 2.7 4.6 2.7.276 0 .5-.224.5-.5s-.224-.5-.5-.5c-1.52 0-2.93-.81-3.7-2.1-.6-1-.6-2.3 0-3.3.77-1.29 2.18-2.1 3.7-2.1.663 0 1.282.23 1.732.62.702.608 1.134 1.983 1.134 3.643v6.278z"/></svg>';
        }

        // Build output - escape all attributes.
        $output = '<a href="' . esc_url( $url ) . '" class="' . esc_attr( $css_class_string ) . '" target="' . esc_attr( $target ) . '" rel="' . esc_attr( $rel ) . '">';
        if ( 'icon' === $style ) {
            $output .= $icon_svg;
        } else {
            if ( 'yes' === $atts['icon'] ) {
                $output .= '<span class="fsb-mastodon-icon-wrapper" style="margin-right:' . $icon_margin . ';">' . $icon_svg . '</span>';
            }
            $output .= '<span class="fsb-mastodon-text">' . esc_html( $text ) . '</span>';
        }
        $output .= '</a>';

        // Add comprehensive inline styles for all button styles.
        $output .= '<style>
            /* Base styles - Mobile First */
            .fsb-mastodon-button {
                display: inline-block;
                text-decoration: none;
                transition: all 0.2s ease;
                box-sizing: border-box;
                word-wrap: break-word;
                max-width: 100%;
            }
            
            .fsb-mastodon-icon-wrapper {
                display: inline-flex;
                align-items: center;
                vertical-align: middle;
            }
            
            .fsb-mastodon-icon {
                flex-shrink: 0;
            }
            
            .fsb-mastodon-text {
                display: inline-block;
                vertical-align: middle;
            }
            
            /* Size variations */
            .fsb-mastodon-size-small {
                font-size: 13px;
                padding: 6px 12px;
            }
            .fsb-mastodon-size-medium {
                font-size: 14px;
                padding: 8px 16px;
            }
            .fsb-mastodon-size-large {
                font-size: 16px;
                padding: 12px 24px;
            }
            
            /* Button Style - Default */
            .fsb-mastodon-button-style {
                background-color: #6364ff;
                color: #fff !important;
                border-radius: 4px;
                font-weight: 500;
            }
            .fsb-mastodon-button-style:hover {
                background-color: #4f50d4;
                color: #fff !important;
                transform: translateY(-1px);
                box-shadow: 0 2px 4px rgba(99, 100, 255, 0.3);
            }
            
            /* Modern Style */
            .fsb-mastodon-modern-style {
                background: linear-gradient(135deg, #6364ff 0%, #8b5cf6 100%);
                color: #fff !important;
                border-radius: 8px;
                font-weight: 600;
                box-shadow: 0 4px 6px rgba(99, 100, 255, 0.2);
            }
            .fsb-mastodon-modern-style:hover {
                transform: translateY(-2px);
                box-shadow: 0 6px 12px rgba(99, 100, 255, 0.3);
                color: #fff !important;
            }
            
            /* Minimal Style */
            .fsb-mastodon-minimal-style {
                background-color: transparent;
                color: #6364ff !important;
                border: 1px solid #6364ff;
                border-radius: 4px;
                font-weight: 500;
            }
            .fsb-mastodon-minimal-style:hover {
                background-color: #6364ff;
                color: #fff !important;
            }
            
            /* Rounded Style */
            .fsb-mastodon-rounded-style {
                background-color: #6364ff;
                color: #fff !important;
                border-radius: 25px;
                font-weight: 500;
            }
            .fsb-mastodon-rounded-style:hover {
                background-color: #4f50d4;
                color: #fff !important;
                transform: scale(1.05);
            }
            
            /* Outline Style */
            .fsb-mastodon-outline-style {
                background-color: transparent;
                color: #6364ff !important;
                border: 2px solid #6364ff;
                border-radius: 4px;
                font-weight: 600;
            }
            .fsb-mastodon-outline-style:hover {
                background-color: #6364ff;
                color: #fff !important;
                border-color: #4f50d4;
            }
            
            /* Gradient Style */
            .fsb-mastodon-gradient-style {
                background: linear-gradient(135deg, #6364ff 0%, #8b5cf6 50%, #ec4899 100%);
                color: #fff !important;
                border-radius: 6px;
                font-weight: 600;
                box-shadow: 0 2px 8px rgba(99, 100, 255, 0.3);
            }
            .fsb-mastodon-gradient-style:hover {
                transform: translateY(-2px);
                box-shadow: 0 4px 12px rgba(99, 100, 255, 0.4);
                color: #fff !important;
            }
            
            /* Pill Style */
            .fsb-mastodon-pill-style {
                background-color: #6364ff;
                color: #fff !important;
                border-radius: 50px;
                font-weight: 500;
                padding-left: 20px;
                padding-right: 20px;
            }
            .fsb-mastodon-pill-style:hover {
                background-color: #4f50d4;
                color: #fff !important;
                transform: scale(1.05);
            }
            
            /* Link Style */
            .fsb-mastodon-link-style {
                color: #6364ff;
                background: transparent;
                padding: 0;
                font-weight: 500;
            }
            .fsb-mastodon-link-style:hover {
                text-decoration: underline;
                color: #4f50d4;
            }
            
            /* Icon Only Style */
            .fsb-mastodon-icon-style {
                display: inline-flex;
                align-items: center;
                justify-content: center;
                color: #6364ff;
                background: transparent;
                padding: 8px;
                border-radius: 50%;
                width: auto;
                height: auto;
            }
            .fsb-mastodon-icon-style:hover {
                background-color: rgba(99, 100, 255, 0.1);
                transform: scale(1.1);
            }
            
            /* Icon Colors */
            .fsb-mastodon-icon-color-default .fsb-mastodon-icon {
                color: #6364ff;
            }
            .fsb-mastodon-icon-color-default:hover {
                background-color: rgba(99, 100, 255, 0.1);
            }
            
            .fsb-mastodon-icon-color-black .fsb-mastodon-icon {
                color: #1a1a1a;
            }
            .fsb-mastodon-icon-color-black:hover {
                background-color: rgba(26, 26, 26, 0.1);
            }
            
            .fsb-mastodon-icon-color-gray .fsb-mastodon-icon {
                color: #666666;
            }
            .fsb-mastodon-icon-color-gray:hover {
                background-color: rgba(102, 102, 102, 0.1);
            }
            
            .fsb-mastodon-icon-color-red .fsb-mastodon-icon {
                color: #e74c3c;
            }
            .fsb-mastodon-icon-color-red:hover {
                background-color: rgba(231, 76, 60, 0.1);
            }
            
            .fsb-mastodon-icon-color-white .fsb-mastodon-icon {
                color: #ffffff;
            }
            .fsb-mastodon-icon-color-white:hover {
                background-color: rgba(255, 255, 255, 0.1);
            }
            
            .fsb-mastodon-icon-color-green .fsb-mastodon-icon {
                color: #27ae60;
            }
            .fsb-mastodon-icon-color-green:hover {
                background-color: rgba(39, 174, 96, 0.1);
            }
            
            .fsb-mastodon-icon-color-purple .fsb-mastodon-icon {
                color: #9b59b6;
            }
            .fsb-mastodon-icon-color-purple:hover {
                background-color: rgba(155, 89, 182, 0.1);
            }
            
            /* White Style - Clean & Minimal */
            .fsb-mastodon-white-style {
                background-color: #ffffff;
                color: #333 !important;
                border: 1px solid #e0e0e0;
                border-radius: 6px;
                font-weight: 500;
                box-shadow: 0 1px 3px rgba(0, 0, 0, 0.1);
            }
            .fsb-mastodon-white-style:hover {
                background-color: #f8f8f8;
                border-color: #d0d0d0;
                color: #333 !important;
                box-shadow: 0 2px 6px rgba(0, 0, 0, 0.15);
                transform: translateY(-1px);
            }
            .fsb-mastodon-white-style .fsb-mastodon-icon {
                color: #333;
            }
            
            /* Black Style - Bold & Modern */
            .fsb-mastodon-black-style {
                background-color: #1a1a1a;
                color: #ffffff !important;
                border: none;
                border-radius: 6px;
                font-weight: 600;
                box-shadow: 0 2px 4px rgba(0, 0, 0, 0.2);
            }
            .fsb-mastodon-black-style:hover {
                background-color: #000000;
                color: #ffffff !important;
                box-shadow: 0 4px 8px rgba(0, 0, 0, 0.3);
                transform: translateY(-1px);
            }
            .fsb-mastodon-black-style .fsb-mastodon-icon {
                color: #ffffff;
            }
            
            /* White Outline Style */
            .fsb-mastodon-white-outline-style {
                background-color: transparent;
                color: #333 !important;
                border: 2px solid #ffffff;
                border-radius: 6px;
                font-weight: 600;
                box-shadow: 0 0 0 1px rgba(0, 0, 0, 0.1);
            }
            .fsb-mastodon-white-outline-style:hover {
                background-color: #ffffff;
                color: #333 !important;
                box-shadow: 0 2px 6px rgba(0, 0, 0, 0.15);
            }
            .fsb-mastodon-white-outline-style .fsb-mastodon-icon {
                color: #333;
            }
            
            /* Black Outline Style */
            .fsb-mastodon-black-outline-style {
                background-color: transparent;
                color: #1a1a1a !important;
                border: 2px solid #1a1a1a;
                border-radius: 6px;
                font-weight: 600;
            }
            .fsb-mastodon-black-outline-style:hover {
                background-color: #1a1a1a;
                color: #ffffff !important;
                box-shadow: 0 2px 6px rgba(0, 0, 0, 0.2);
            }
            .fsb-mastodon-black-outline-style .fsb-mastodon-icon {
                color: #1a1a1a;
            }
            .fsb-mastodon-black-outline-style:hover .fsb-mastodon-icon {
                color: #ffffff;
            }
            
            /* White Minimal Style */
            .fsb-mastodon-white-minimal-style {
                background-color: #f8f8f8;
                color: #333 !important;
                border: none;
                border-radius: 4px;
                font-weight: 500;
            }
            .fsb-mastodon-white-minimal-style:hover {
                background-color: #eeeeee;
                color: #333 !important;
            }
            .fsb-mastodon-white-minimal-style .fsb-mastodon-icon {
                color: #666;
            }
            
            /* Black Minimal Style */
            .fsb-mastodon-black-minimal-style {
                background-color: #2a2a2a;
                color: #ffffff !important;
                border: none;
                border-radius: 4px;
                font-weight: 500;
            }
            .fsb-mastodon-black-minimal-style:hover {
                background-color: #3a3a3a;
                color: #ffffff !important;
            }
            .fsb-mastodon-black-minimal-style .fsb-mastodon-icon {
                color: #ffffff;
            }
            
            /* Mobile Responsive */
            @media (max-width: 768px) {
                .fsb-mastodon-button {
                    width: 100%;
                    text-align: center;
                    display: block;
                    margin: 5px 0;
                }
                .fsb-mastodon-size-small {
                    padding: 8px 16px;
                    font-size: 14px;
                }
                .fsb-mastodon-size-medium {
                    padding: 10px 20px;
                    font-size: 15px;
                }
                .fsb-mastodon-size-large {
                    padding: 12px 24px;
                    font-size: 16px;
                }
            }
            
            @media (max-width: 480px) {
                .fsb-mastodon-button {
                    font-size: 14px !important;
                }
                .fsb-mastodon-icon {
                    width: 18px !important;
                    height: 18px !important;
                }
            }
        </style>';

        return $output;
    }

    /**
     * Mastodon feed shortcode.
     *
     * @param array  $atts    Shortcode attributes.
     * @param string $content Shortcode content.
     * @return string
     */
    public function mastodon_feed_shortcode( $atts, $content = null ) {
        $this->debug_log( 'mastodon_feed shortcode called with atts: ' . wp_json_encode( $atts ) );
        
        $atts = shortcode_atts(
            array(
                'username'       => '',
                'tag'            => '',
                'count'          => 5,
                'instance'       => Fifth_Social_Bot_Core::get_instance_url(),
                'cache_time'     => 300, // 5 minutes.
                'exclude_replies' => 'false', // true/false - exclude replies from feed.
                'exclude_reblogs' => 'false', // true/false - exclude reblogs from feed.
                'only_media'     => 'false', // true/false - show only posts with media.
                'layout'         => 'default', // default, compact, grid.
                'custom_css'     => '', // Custom CSS class or inline styles.
            ),
            $atts,
            'mastodon_feed'
        );

        $username = trim( $atts['username'] );
        $tag      = trim( $atts['tag'] );
        $count    = max( 1, min( 10, (int) $atts['count'] ) ); // Maximum 10 posts allowed.
        
        // Normalize instance URL - ensure it has protocol and trailing slash.
        $instance = trim( $atts['instance'] );
        $instance = sanitize_text_field( $instance );
        if ( empty( $instance ) ) {
            $instance = Fifth_Social_Bot_Core::get_instance_url();
        }
        
        // Fix common URL mistakes - correct social5th.ro to social.5th.ro.
        $instance = str_replace( 'social5th.ro', 'social.5th.ro', $instance );
        $instance = str_replace( 'social5th', 'social.5th', $instance );
        
        // Add https:// if missing.
        if ( ! preg_match( '#^https?://#', $instance ) ) {
            $instance = 'https://' . $instance;
        }
        
        // Validate URL format for security.
        $instance = esc_url_raw( $instance );
        if ( empty( $instance ) ) {
            $this->debug_log( 'mastodon_feed: Invalid instance URL format', 'error' );
            return '<p>' . esc_html__( 'Error: Invalid Mastodon instance URL.', '5th-social-bot' ) . '</p>';
        }
        
        // Extract domain to check for common typos.
        $instance_domain = preg_replace( '#^https?://#', '', $instance );
        $instance_domain = rtrim( $instance_domain, '/' );
        
        // Additional corrections for common typos.
        $corrections = array(
            'social5th.ro' => 'social.5th.ro',
            'social5th' => 'social.5th',
        );
        
        foreach ( $corrections as $wrong => $correct ) {
            if ( strpos( $instance_domain, $wrong ) !== false ) {
                $instance_domain = str_replace( $wrong, $correct, $instance_domain );
                // Rebuild instance URL.
                $protocol = preg_match( '#^https://#', $instance ) ? 'https://' : 'http://';
                $instance = $protocol . $instance_domain;
                break;
            }
        }
        
        $instance = trailingslashit( $instance );

        // Use username from settings if not provided.
        if ( empty( $username ) ) {
            $username = self::get_mastodon_username();
        }

        if ( empty( $username ) && empty( $tag ) ) {
            return '<p>' . esc_html__( 'Mastodon username or tag required.', '5th-social-bot' ) . '</p>';
        }

        // Remove @ if present.
        $username = ltrim( $username, '@' );

        // Build API URL.
        if ( ! empty( $tag ) ) {
            // Tag feed - public API, no auth needed.
            // Sanitize tag to prevent injection.
            $tag = sanitize_text_field( $tag );
            $api_url = $instance . 'api/v1/timelines/tag/' . rawurlencode( $tag ) . '?limit=' . absint( $count );
        } else {
            // User feed - need to lookup account first.
            // For cross-instance lookup, use format: username@instance.domain
            // Extract instance domain from instance URL (already normalized above with corrections).
            $instance_domain = preg_replace( '#^https?://#', '', $instance );
            $instance_domain = rtrim( $instance_domain, '/' );
            
            // Sanitize username to prevent injection.
            $username = sanitize_text_field( $username );
            
            // Build acct parameter - if username doesn't contain @, add instance domain.
            if ( strpos( $username, '@' ) === false ) {
                $acct = $username . '@' . $instance_domain;
            } else {
                $acct = $username; // Already has @instance format.
            }
            
            // Sanitize acct before encoding.
            $acct = sanitize_text_field( $acct );
            $api_url = $instance . 'api/v1/accounts/lookup?acct=' . rawurlencode( $acct );
        }

        // Cache key.
        $cache_key = 'fsb_feed_' . md5( $api_url . $count );
        $feed_data = get_transient( $cache_key );

        if ( false === $feed_data ) {
            $this->debug_log( 'mastodon_feed: Cache miss, fetching from API: ' . $api_url );
            // Fetch from API.
            if ( ! empty( $tag ) ) {
                // Tag feed - direct API call (public endpoint).
                $response = wp_remote_get(
                    $api_url,
                    array(
                        'timeout' => 15,
                        'headers' => array(
                            'User-Agent' => 'WordPress/5th-Social-Bot-Plugin',
                        ),
                    )
                );
            } else {
                // User feed - first get account ID, then statuses.
                // This is a public API endpoint - no authentication needed for public accounts.
                $lookup_response = wp_remote_get(
                    $api_url,
                    array(
                        'timeout' => 15,
                        'headers' => array(
                            'User-Agent' => 'WordPress/5th-Social-Bot-Plugin',
                        ),
                    )
                );

                if ( is_wp_error( $lookup_response ) ) {
                    return '<p>' . esc_html__( 'Error fetching Mastodon feed.', '5th-social-bot' ) . '</p>';
                }

                $lookup_code = wp_remote_retrieve_response_code( $lookup_response );
                if ( $lookup_code < 200 || $lookup_code >= 300 ) {
                    return '<p>' . esc_html__( 'Error fetching Mastodon feed.', '5th-social-bot' ) . '</p>';
                }

                $lookup_body = wp_remote_retrieve_body( $lookup_response );
                $lookup_data = json_decode( $lookup_body, true );

                if ( ! is_array( $lookup_data ) || ! isset( $lookup_data['id'] ) ) {
                    return '<p>' . esc_html__( 'Mastodon account not found.', '5th-social-bot' ) . '</p>';
                }

                $account_id = isset( $lookup_data['id'] ) ? absint( $lookup_data['id'] ) : 0;
                if ( $account_id <= 0 ) {
                    $this->debug_log( 'mastodon_feed: Invalid account ID from lookup', 'error' );
                    return '<p>' . esc_html__( 'Mastodon account not found.', '5th-social-bot' ) . '</p>';
                }
                
                // Get public statuses with optional filters.
                // This endpoint is public and doesn't require authentication for public posts.
                $exclude_replies = ( 'true' === strtolower( $atts['exclude_replies'] ) ) ? 'true' : 'false';
                $exclude_reblogs = ( 'true' === strtolower( $atts['exclude_reblogs'] ) ) ? 'true' : 'false';
                $only_media      = ( 'true' === strtolower( $atts['only_media'] ) ) ? 'true' : 'false';
                
                $api_url = $instance . 'api/v1/accounts/' . $account_id . '/statuses?limit=' . absint( $count );
                $api_url .= '&exclude_replies=' . esc_attr( $exclude_replies );
                $api_url .= '&exclude_reblogs=' . esc_attr( $exclude_reblogs );
                if ( 'true' === $only_media ) {
                    $api_url .= '&only_media=true';
                }

                $response = wp_remote_get(
                    $api_url,
                    array(
                        'timeout' => 15,
                        'headers' => array(
                            'User-Agent' => 'WordPress/5th-Social-Bot-Plugin',
                        ),
                    )
                );
            }

            if ( is_wp_error( $response ) ) {
                return '<p>' . esc_html__( 'Error fetching Mastodon feed.', '5th-social-bot' ) . '</p>';
            }

            $code = wp_remote_retrieve_response_code( $response );
            if ( $code < 200 || $code >= 300 ) {
                return '<p>' . esc_html__( 'Error fetching Mastodon feed.', '5th-social-bot' ) . '</p>';
            }

            $body = wp_remote_retrieve_body( $response );
            $feed_data = json_decode( $body, true );

            if ( ! is_array( $feed_data ) ) {
                return '<p>' . esc_html__( 'Invalid feed data.', '5th-social-bot' ) . '</p>';
            }

            // Cache the result.
            set_transient( $cache_key, $feed_data, (int) $atts['cache_time'] );
            $this->debug_log( 'mastodon_feed: Fetched ' . count( $feed_data ) . ' posts, cached for ' . $atts['cache_time'] . ' seconds' );
        } else {
            $this->debug_log( 'mastodon_feed: Using cached data (' . count( $feed_data ) . ' posts)' );
        }

        if ( empty( $feed_data ) ) {
            $this->debug_log( 'mastodon_feed: No posts found', 'error' );
            return '<p>' . esc_html__( 'No posts found.', '5th-social-bot' ) . '</p>';
        }
        
        $this->debug_log( 'mastodon_feed: Rendering ' . count( $feed_data ) . ' posts with layout: ' . $layout );

        // Filter feed data if needed.
        if ( 'true' === strtolower( $atts['only_media'] ) ) {
            $feed_data = array_filter(
                $feed_data,
                function( $status ) {
                    return ! empty( $status['media_attachments'] ) && is_array( $status['media_attachments'] ) && ! empty( $status['media_attachments'] );
                }
            );
            $feed_data = array_values( $feed_data ); // Re-index array.
        }
        
        // Layout class.
        $layout = sanitize_key( $atts['layout'] );
        if ( ! in_array( $layout, array( 'default', 'compact', 'grid' ), true ) ) {
            $layout = 'default';
        }
        
        // Custom CSS class if provided.
        $custom_css_class = '';
        if ( ! empty( $atts['custom_css'] ) ) {
            $custom_css_class = ' ' . sanitize_html_class( $atts['custom_css'] );
        }
        
        // Render feed.
        $output = '<div class="fsb-mastodon-feed fsb-mastodon-feed-layout-' . esc_attr( $layout ) . $custom_css_class . '">';
        $output .= '<style>
            .fsb-mastodon-feed {
                max-width: 100%;
                font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, Oxygen-Sans, Ubuntu, Cantarell, "Helvetica Neue", sans-serif;
            }
            .fsb-mastodon-feed-item {
                margin-bottom: 20px;
                padding: 20px;
                border: 1px solid #e0e0e0;
                border-radius: 12px;
                background: #fff;
                box-shadow: 0 1px 3px rgba(0, 0, 0, 0.05);
                transition: all 0.3s ease;
            }
            .fsb-mastodon-feed-item:hover {
                box-shadow: 0 4px 12px rgba(0, 0, 0, 0.1);
                transform: translateY(-2px);
            }
            
            /* Compact Layout */
            .fsb-mastodon-feed-layout-compact .fsb-mastodon-feed-item {
                padding: 12px;
                margin-bottom: 12px;
                border-radius: 8px;
            }
            .fsb-mastodon-feed-layout-compact .fsb-mastodon-feed-item-avatar {
                width: 36px;
                height: 36px;
                margin-right: 8px;
            }
            .fsb-mastodon-feed-layout-compact .fsb-mastodon-feed-item-header-info strong {
                font-size: 13px;
            }
            .fsb-mastodon-feed-layout-compact .fsb-mastodon-feed-item-header-info span {
                font-size: 11px;
            }
            .fsb-mastodon-feed-layout-compact .fsb-mastodon-feed-item-content {
                font-size: 13px;
                margin-bottom: 8px;
            }
            .fsb-mastodon-feed-layout-compact .fsb-mastodon-feed-item-meta {
                font-size: 11px;
                padding-top: 8px;
            }
            
            /* Grid Layout */
            .fsb-mastodon-feed-layout-grid {
                display: grid !important;
                grid-template-columns: repeat(auto-fill, minmax(300px, 1fr)) !important;
                gap: 20px !important;
                width: 100% !important;
            }
            .fsb-mastodon-feed-layout-grid .fsb-mastodon-feed-item {
                margin-bottom: 0 !important;
                width: 100% !important;
                box-sizing: border-box !important;
            }
            @media (max-width: 768px) {
                .fsb-mastodon-feed-layout-grid {
                    grid-template-columns: 1fr !important;
                    gap: 15px !important;
                }
            }
            @media (min-width: 769px) and (max-width: 1024px) {
                .fsb-mastodon-feed-layout-grid {
                    grid-template-columns: repeat(auto-fill, minmax(250px, 1fr)) !important;
                }
            }
            .fsb-mastodon-feed-item-header {
                display: flex;
                align-items: center;
                margin-bottom: 12px;
            }
            .fsb-mastodon-feed-item-avatar {
                width: 48px;
                height: 48px;
                border-radius: 50%;
                margin-right: 12px;
                flex-shrink: 0;
                object-fit: cover;
            }
            .fsb-mastodon-feed-item-header-info {
                flex: 1;
                min-width: 0;
            }
            .fsb-mastodon-feed-item-header-info strong {
                display: block;
                font-size: 15px;
                font-weight: 600;
                color: #333;
                margin-bottom: 2px;
            }
            .fsb-mastodon-feed-item-header-info span {
                display: block;
                font-size: 13px;
                color: #666;
            }
            .fsb-mastodon-feed-item-content {
                line-height: 1.6;
                margin-bottom: 12px;
                color: #333;
                font-size: 15px;
                word-wrap: break-word;
            }
            .fsb-mastodon-feed-item-content p {
                margin: 0 0 10px 0;
            }
            .fsb-mastodon-feed-item-content p:last-child {
                margin-bottom: 0;
            }
            .fsb-mastodon-feed-item-content a {
                color: #6364ff;
                text-decoration: none;
            }
            .fsb-mastodon-feed-item-content a:hover {
                text-decoration: underline;
            }
            .fsb-mastodon-feed-item-media {
                margin-top: 12px;
                margin-bottom: 12px;
            }
            .fsb-mastodon-feed-item-media img {
                max-width: 100%;
                height: auto;
                border-radius: 8px;
                display: block;
            }
            .fsb-mastodon-feed-item-meta {
                font-size: 13px;
                color: #666;
                padding-top: 12px;
                border-top: 1px solid #f0f0f0;
                display: flex;
                align-items: center;
                gap: 8px;
            }
            .fsb-mastodon-feed-item-meta span {
                color: #999;
            }
            .fsb-mastodon-feed-item-meta a {
                color: #6364ff;
                text-decoration: none;
                font-weight: 500;
            }
            .fsb-mastodon-feed-item-meta a:hover {
                text-decoration: underline;
            }
            /* Mobile Responsive */
            @media (max-width: 768px) {
                .fsb-mastodon-feed-item {
                    padding: 15px;
                    margin-bottom: 15px;
                }
                .fsb-mastodon-feed-item-avatar {
                    width: 40px;
                    height: 40px;
                    margin-right: 10px;
                }
                .fsb-mastodon-feed-item-header-info strong {
                    font-size: 14px;
                }
                .fsb-mastodon-feed-item-header-info span {
                    font-size: 12px;
                }
                .fsb-mastodon-feed-item-content {
                    font-size: 14px;
                }
                .fsb-mastodon-feed-item-meta {
                    font-size: 12px;
                    flex-wrap: wrap;
                }
            }
        </style>';

        foreach ( $feed_data as $status ) {
            if ( ! is_array( $status ) ) {
                continue;
            }

            $status_url = isset( $status['url'] ) ? esc_url( $status['url'] ) : '';
            $content    = isset( $status['content'] ) ? wp_kses_post( $status['content'] ) : '';
            $created_at = isset( $status['created_at'] ) ? $status['created_at'] : '';
            $account    = isset( $status['account'] ) ? $status['account'] : array();
            $media_attachments = isset( $status['media_attachments'] ) ? $status['media_attachments'] : array();
            $avatar     = isset( $account['avatar'] ) ? esc_url( $account['avatar'] ) : '';
            $display_name = isset( $account['display_name'] ) ? esc_html( $account['display_name'] ) : '';
            $username_display = isset( $account['acct'] ) ? esc_html( '@' . $account['acct'] ) : '';
            $account_url = isset( $account['url'] ) ? esc_url( $account['url'] ) : '';

            // Format date - use relative time if recent, otherwise full date.
            $date_formatted = '';
            $relative_time = '';
            if ( ! empty( $created_at ) ) {
                // Sanitize date string before parsing.
                $created_at = sanitize_text_field( $created_at );
                $timestamp = strtotime( $created_at );
                $now = time();
                $diff = $now - $timestamp;
                
                if ( $diff < 3600 ) {
                    // Less than 1 hour ago.
                    $minutes = floor( $diff / 60 );
                    // translators: %d: number of minutes.
                    $relative_time = $minutes <= 1 ? __( 'just now', '5th-social-bot' ) : sprintf( __( '%d minutes ago', '5th-social-bot' ), $minutes );
                } elseif ( $diff < 86400 ) {
                    // Less than 24 hours ago.
                    $hours = floor( $diff / 3600 );
                    // translators: %d: number of hours.
                    $relative_time = sprintf( __( '%d hours ago', '5th-social-bot' ), $hours );
                } elseif ( $diff < 604800 ) {
                    // Less than a week ago.
                    $days = floor( $diff / 86400 );
                    // translators: %d: number of days.
                    $relative_time = sprintf( __( '%d days ago', '5th-social-bot' ), $days );
                }
                
                $date_formatted = date_i18n( get_option( 'date_format' ) . ' ' . get_option( 'time_format' ), $timestamp );
            }

            $output .= '<div class="fsb-mastodon-feed-item">';
            
            // Header with avatar and user info.
            $output .= '<div class="fsb-mastodon-feed-item-header">';
            if ( ! empty( $avatar ) ) {
                $avatar_link = ! empty( $account_url ) ? $account_url : ( ! empty( $status_url ) ? $status_url : '' );
                if ( ! empty( $avatar_link ) ) {
                    $output .= '<a href="' . esc_url( $avatar_link ) . '" target="_blank" rel="noopener noreferrer">';
                }
                $output .= '<img src="' . esc_url( $avatar ) . '" alt="' . esc_attr( $display_name ?: $username_display ) . '" class="fsb-mastodon-feed-item-avatar" />';
                if ( ! empty( $avatar_link ) ) {
                    $output .= '</a>';
                }
            }
            $output .= '<div class="fsb-mastodon-feed-item-header-info">';
            if ( ! empty( $display_name ) ) {
                $output .= '<strong>' . esc_html( $display_name ) . '</strong>';
            }
            $output .= '<span>' . esc_html( $username_display ) . '</span>';
            $output .= '</div>';
            $output .= '</div>';

            // Content.
            $output .= '<div class="fsb-mastodon-feed-item-content">' . $content . '</div>';

            // Media attachments (images).
            if ( ! empty( $media_attachments ) && is_array( $media_attachments ) ) {
                $output .= '<div class="fsb-mastodon-feed-item-media">';
                foreach ( $media_attachments as $media ) {
                    if ( isset( $media['type'] ) && 'image' === $media['type'] && isset( $media['url'] ) ) {
                        $media_url = isset( $media['url'] ) ? $media['url'] : '';
                        $media_preview = isset( $media['preview_url'] ) ? $media['preview_url'] : $media_url;
                        $media_url = esc_url( $media_url );
                        $media_preview = esc_url( $media_preview );
                        $output .= '<a href="' . esc_url( $media_url ) . '" target="_blank" rel="noopener noreferrer">';
                        $output .= '<img src="' . esc_url( $media_preview ) . '" alt="" />';
                        $output .= '</a>';
                    }
                }
                $output .= '</div>';
            }

            // Meta (date and link).
            $output .= '<div class="fsb-mastodon-feed-item-meta">';
            if ( ! empty( $relative_time ) ) {
                $output .= '<span title="' . esc_attr( $date_formatted ) . '">' . esc_html( $relative_time ) . '</span>';
            } elseif ( ! empty( $date_formatted ) ) {
                $output .= '<span>' . esc_html( $date_formatted ) . '</span>';
            }
            if ( ! empty( $status_url ) ) {
                $output .= '<span>·</span>';
                $output .= '<a href="' . esc_url( $status_url ) . '" target="_blank" rel="noopener noreferrer">' . esc_html__( 'View on Mastodon', '5th-social-bot' ) . '</a>';
            }
            $output .= '</div>';

            $output .= '</div>';
        }

        $output .= '</div>';

        return $output;
    }
}

