<?php
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Auto-Boost functionality: automatically boost posts on a second Mastodon instance.
 */
class Fifth_Social_Bot_Boost {

    public function __construct() {
        add_action( 'fifth_social_bot_process_boost_queue_event', array( __CLASS__, 'process_boost_queue' ) );
    }

    /**
     * Get boost queue (list of post IDs waiting for boost).
     *
     * @return array
     */
    public static function get_boost_queue() {
        $queue = get_option( 'fifth_social_bot_boost_queue', array() );
        if ( ! is_array( $queue ) ) {
            $queue = array();
        }
        $queue = array_filter( array_map( 'intval', $queue ) );
        return array_values( array_unique( $queue ) );
    }

    /**
     * Save boost queue.
     *
     * @param array $queue Queue.
     */
    public static function set_boost_queue( $queue ) {
        if ( ! is_array( $queue ) ) {
            $queue = array();
        }
        $queue = array_filter( array_map( 'intval', $queue ) );
        $queue = array_values( array_unique( $queue ) );
        update_option( 'fifth_social_bot_boost_queue', $queue, false );
    }

    /**
     * Add post to boost queue.
     *
     * @param int $post_id Post ID.
     */
    public static function enqueue_boost( $post_id ) {
        $post_id = (int) $post_id;
        if ( $post_id <= 0 ) {
            return;
        }

        $queue = self::get_boost_queue();
        if ( in_array( $post_id, $queue, true ) ) {
            return; // Already queued.
        }

        // Check if already boosted.
        $boost_url = get_post_meta( $post_id, '_fifth_social_bot_boost_url', true );
        if ( ! empty( $boost_url ) ) {
            return; // Already boosted.
        }

        $queue[] = $post_id;
        self::set_boost_queue( $queue );
        update_post_meta( $post_id, '_fifth_social_bot_boost_queued', '1' );
        self::schedule_boost_queue();
    }

    /**
     * Schedule boost queue processing event.
     */
    public static function schedule_boost_queue() {
        if ( ! wp_next_scheduled( 'fifth_social_bot_process_boost_queue_event' ) ) {
            wp_schedule_single_event( time() + 60, 'fifth_social_bot_process_boost_queue_event' );
        }
    }

    /**
     * Process boost queue: try to boost 1 post, respecting rate limits.
     */
    public static function process_boost_queue() {
        $queue = self::get_boost_queue();
        if ( empty( $queue ) ) {
            return;
        }

        $settings = Fifth_Social_Bot_Core::get_settings();
        if ( empty( $settings['boost_enabled'] ) ) {
            // Boost disabled, clear queue.
            self::set_boost_queue( array() );
            return;
        }

        // Check rate limits.
        $rate_limit_check = self::check_rate_limits();
        if ( ! $rate_limit_check['allowed'] ) {
            // Rate limited, reschedule for later.
            if ( ! empty( $rate_limit_check['next_boost_time'] ) ) {
                $next_time = (int) $rate_limit_check['next_boost_time'];
                $current_time = time();
                if ( $next_time > $current_time ) {
                    wp_schedule_single_event( $next_time, 'fifth_social_bot_process_boost_queue_event' );
                } else {
                    // Try again in 5 minutes.
                    wp_schedule_single_event( time() + 300, 'fifth_social_bot_process_boost_queue_event' );
                }
            } else {
                // Try again in 5 minutes.
                wp_schedule_single_event( time() + 300, 'fifth_social_bot_process_boost_queue_event' );
            }
            return;
        }

        // Process first post in queue.
        $post_id = $queue[0];
        $remaining = array_slice( $queue, 1 );

        // Get Mastodon URL for this post.
        $mastodon_url = get_post_meta( $post_id, '_fifth_social_bot_mastodon_url', true );
        if ( empty( $mastodon_url ) ) {
            // No Mastodon URL yet, keep in queue.
            self::set_boost_queue( $queue );
            // Reschedule for later.
            wp_schedule_single_event( time() + 300, 'fifth_social_bot_process_boost_queue_event' );
            return;
        }

        // Try to boost.
        $boost_result = self::boost_post( $mastodon_url, $post_id );
        
        if ( $boost_result['success'] && ! empty( $boost_result['url'] ) ) {
            // Success - remove from queue.
            delete_post_meta( $post_id, '_fifth_social_bot_boost_queued' );
            delete_post_meta( $post_id, '_fifth_social_bot_boost_next_time' );
            self::set_boost_queue( $remaining );
        } else {
            // Failed or rate limited - keep in queue.
            self::set_boost_queue( $queue );
            if ( ! empty( $rate_limit_check['next_boost_time'] ) ) {
                $next_time = (int) $rate_limit_check['next_boost_time'];
                wp_schedule_single_event( $next_time, 'fifth_social_bot_process_boost_queue_event' );
            } else {
                // Get minimum interval from settings.
                $settings = Fifth_Social_Bot_Core::get_settings();
                $min_interval_minutes = isset( $settings['boost_min_interval'] ) ? (int) $settings['boost_min_interval'] : 30;
                $min_interval_minutes = max( 30, min( 120, $min_interval_minutes ) ); // 30 min to 2h
                $min_interval_seconds = $min_interval_minutes * 60;
                
                // Try again after minimum interval.
                wp_schedule_single_event( time() + $min_interval_seconds, 'fifth_social_bot_process_boost_queue_event' );
            }
            return;
        }

        // Schedule next boost if there are more in queue.
        if ( ! empty( $remaining ) ) {
            // Get minimum interval from settings.
            $settings = Fifth_Social_Bot_Core::get_settings();
            $min_interval_minutes = isset( $settings['boost_min_interval'] ) ? (int) $settings['boost_min_interval'] : 30;
            $min_interval_minutes = max( 30, min( 120, $min_interval_minutes ) ); // 30 min to 2h
            $min_interval_seconds = $min_interval_minutes * 60;
            
            // Schedule next boost after minimum interval.
            wp_schedule_single_event( time() + $min_interval_seconds, 'fifth_social_bot_process_boost_queue_event' );
        }
    }

    /**
     * Check rate limits for boost.
     *
     * @return array { allowed: bool, message: string, next_boost_time: int }
     */
    protected static function check_rate_limits() {
        $result = array(
            'allowed'        => false,
            'message'        => '',
            'next_boost_time' => 0,
        );

        // Get settings for rate limits.
        $settings = Fifth_Social_Bot_Core::get_settings();
        $max_per_day = isset( $settings['boost_max_per_day'] ) ? (int) $settings['boost_max_per_day'] : 10;
        $min_interval_minutes = isset( $settings['boost_min_interval'] ) ? (int) $settings['boost_min_interval'] : 30;
        
        // Validate and clamp values.
        $max_per_day = max( 1, min( 10, $max_per_day ) );
        $min_interval_minutes = max( 30, min( 120, $min_interval_minutes ) ); // 30 min to 2h
        
        $min_interval_seconds = $min_interval_minutes * 60;

        // Get boost stats from options.
        $boost_stats = get_option( 'fifth_social_bot_boost_stats', array() );
        $today = gmdate( 'Y-m-d' );
        $last_boost_date = isset( $boost_stats['last_boost_date'] ) ? $boost_stats['last_boost_date'] : '';
        $boost_count_today = isset( $boost_stats['boost_count_today'] ) ? (int) $boost_stats['boost_count_today'] : 0;
        $last_boost_time = isset( $boost_stats['last_boost_time'] ) ? (int) $boost_stats['last_boost_time'] : 0;

        // Reset count if it's a new day.
        if ( $today !== $last_boost_date ) {
            $boost_count_today = 0;
            $boost_stats['boost_count_today'] = 0;
            $boost_stats['last_boost_date'] = $today;
            update_option( 'fifth_social_bot_boost_stats', $boost_stats );
        }

        // Check daily limit.
        if ( $boost_count_today >= $max_per_day ) {
            $result['message'] = sprintf(
                /* translators: %d: maximum number of boosts per day */
                __( 'Daily boost limit reached (%d boosts per day). Please try again tomorrow.', '5th-social-bot' ),
                $max_per_day
            );
            return $result;
        }

        // Check minimum interval.
        $current_time = time();
        $time_since_last_boost = $current_time - $last_boost_time;

        if ( $last_boost_time > 0 && $time_since_last_boost < $min_interval_seconds ) {
            $remaining_seconds = $min_interval_seconds - $time_since_last_boost;
            $remaining_minutes = ceil( $remaining_seconds / 60 );
            $result['next_boost_time'] = $last_boost_time + $min_interval_seconds;
            $result['message'] = sprintf(
                /* translators: %d: number of minutes */
                __( 'Minimum interval not reached. Please wait %d more minute(s) before the next boost.', '5th-social-bot' ),
                $remaining_minutes
            );
            return $result;
        }

        $result['allowed'] = true;
        return $result;
    }

    /**
     * Update boost statistics after a successful boost.
     */
    protected static function update_boost_stats() {
        $boost_stats = get_option( 'fifth_social_bot_boost_stats', array() );
        $today = gmdate( 'Y-m-d' );
        $last_boost_date = isset( $boost_stats['last_boost_date'] ) ? $boost_stats['last_boost_date'] : '';
        $boost_count_today = isset( $boost_stats['boost_count_today'] ) ? (int) $boost_stats['boost_count_today'] : 0;

        // Reset count if it's a new day.
        if ( $today !== $last_boost_date ) {
            $boost_count_today = 0;
        }

        $boost_stats['last_boost_date'] = $today;
        $boost_stats['boost_count_today'] = $boost_count_today + 1;
        $boost_stats['last_boost_time'] = time();
        update_option( 'fifth_social_bot_boost_stats', $boost_stats );
    }

    /**
     * Validate and normalize instance URL.
     *
     * @param string $instance_url Raw instance URL.
     * @return array { valid: bool, url: string, message: string }
     */
    public static function validate_instance_url( $instance_url ) {
        $result = array(
            'valid'   => false,
            'url'     => '',
            'message' => '',
        );

        if ( empty( $instance_url ) ) {
            $result['message'] = __( 'Instance URL is empty.', '5th-social-bot' );
            return $result;
        }

        // Normalize instance URL.
        $instance = trim( $instance_url );
        $instance = sanitize_text_field( $instance );
        $instance = rtrim( $instance, '/' );
        if ( ! preg_match( '#^https?://#', $instance ) ) {
            $instance = 'https://' . $instance;
        }

        // Parse URL to validate hostname.
        $parsed = wp_parse_url( $instance );
        if ( false === $parsed || empty( $parsed['host'] ) ) {
            $result['message'] = __( 'Invalid instance URL format. Please enter a valid URL like https://mastodon.social or https://comunicat.ro', '5th-social-bot' );
            return $result;
        }

        $host = $parsed['host'];

        // Check if hostname has a valid TLD (contains at least one dot and has a domain extension).
        // This prevents URLs like "https://comunicat" which cannot be resolved.
        if ( ! preg_match( '/^[a-z0-9]([a-z0-9\-]{0,61}[a-z0-9])?(\.[a-z0-9]([a-z0-9\-]{0,61}[a-z0-9])?)+$/i', $host ) ) {
            $result['message'] = sprintf(
                /* translators: %s: the invalid hostname */
                __( 'Invalid hostname "%s". The instance URL must include a valid domain (e.g., https://comunicat.ro or https://mastodon.social). Please check your configuration.', '5th-social-bot' ),
                esc_html( $host )
            );
            return $result;
        }

        // Use esc_url_raw to ensure proper URL encoding.
        $instance = esc_url_raw( $instance );
        if ( empty( $instance ) ) {
            $result['message'] = __( 'Invalid instance URL format.', '5th-social-bot' );
            return $result;
        }

        $result['valid'] = true;
        $result['url']   = $instance;
        return $result;
    }

    /**
     * Boost a Mastodon post on the second instance.
     *
     * @param string $mastodon_url URL of the post on social.5th.ro.
     * @param int    $post_id      WordPress post ID.
     * @return array { success: bool, url: string, message: string }
     */
    public static function boost_post( $mastodon_url, $post_id ) {
        $result = array(
            'success' => false,
            'url'     => '',
            'message' => '',
        );

        $settings = Fifth_Social_Bot_Core::get_settings();

        // Check if boost is enabled and configured.
        if ( empty( $settings['boost_enabled'] ) ) {
            $result['message'] = __( 'Auto-boost is disabled.', '5th-social-bot' );
            return $result;
        }

        // Check rate limits.
        $rate_limit_check = self::check_rate_limits();
        if ( ! $rate_limit_check['allowed'] ) {
            $result['message'] = $rate_limit_check['message'];
            // Mark post as queued for boost if rate limited.
            update_post_meta( $post_id, '_fifth_social_bot_boost_queued', '1' );
            if ( ! empty( $rate_limit_check['next_boost_time'] ) ) {
                update_post_meta( $post_id, '_fifth_social_bot_boost_next_time', $rate_limit_check['next_boost_time'] );
            }
            Fifth_Social_Bot_Logger::log(
                'Boost rate limited for post ID ' . $post_id . ': ' . $rate_limit_check['message'],
                'info'
            );
            return $result;
        }

        // Check if primary account is set (required).
        if ( empty( $settings['access_token'] ) ) {
            $result['message'] = __( 'Primary Mastodon account (social.5th.ro) is not configured.', '5th-social-bot' );
            return $result;
        }

        // Check if second instance is configured.
        if ( empty( $settings['boost_instance'] ) || empty( $settings['boost_access_token'] ) ) {
            $result['message'] = __( 'Second Mastodon instance is not configured.', '5th-social-bot' );
            return $result;
        }

        // Validate and normalize instance URL.
        $url_validation = self::validate_instance_url( $settings['boost_instance'] );
        if ( ! $url_validation['valid'] ) {
            $result['message'] = $url_validation['message'];
            Fifth_Social_Bot_Logger::log(
                'Boost failed for post ID ' . $post_id . ': ' . $url_validation['message'],
                'error'
            );
            return $result;
        }
        $instance = $url_validation['url'];

        // Clean access token.
        $access_token = trim( $settings['boost_access_token'] );
        $access_token = preg_replace( '/\s+/', '', $access_token );
        if ( empty( $access_token ) ) {
            $result['message'] = __( 'Second instance access token is empty.', '5th-social-bot' );
            return $result;
        }

        // For cross-instance boost, we need to find the post on the second instance first.
        // Search for the post using the URL from social.5th.ro.
        // Note: The `resolve=true` parameter is crucial - it tells Mastodon to resolve external URLs
        // and find federated posts. Without it, search won't find posts from other instances.
        $search_url = trailingslashit( $instance ) . 'api/v2/search?q=' . rawurlencode( $mastodon_url ) . '&type=statuses&limit=1&resolve=true';
        
        Fifth_Social_Bot_Logger::log( 'Searching for post on second instance: ' . $search_url, 'debug' );
        Fifth_Social_Bot_Logger::log( 'Post URL from primary instance: ' . $mastodon_url, 'debug' );
        
        $search_response = wp_remote_get(
            $search_url,
            array(
                'timeout' => 15,
                'headers' => array(
                    'Authorization' => 'Bearer ' . $access_token,
                ),
            )
        );

        $status_id = null;
        if ( ! is_wp_error( $search_response ) ) {
            $search_code = wp_remote_retrieve_response_code( $search_response );
            Fifth_Social_Bot_Logger::log( 'Search API response code: ' . $search_code, 'debug' );
            
            if ( $search_code >= 200 && $search_code < 300 ) {
                $search_body = wp_remote_retrieve_body( $search_response );
                $search_data = json_decode( $search_body, true );
                Fifth_Social_Bot_Logger::log( 'Search API response: ' . substr( $search_body, 0, 500 ), 'debug' );
                
                if ( is_array( $search_data ) && isset( $search_data['statuses'] ) && is_array( $search_data['statuses'] ) && ! empty( $search_data['statuses'] ) ) {
                    $found_status = $search_data['statuses'][0];
                    if ( isset( $found_status['id'] ) ) {
                        $status_id = absint( $found_status['id'] );
                        Fifth_Social_Bot_Logger::log( 'Found status ID ' . $status_id . ' on second instance via search', 'info' );
                    }
                } else {
                    Fifth_Social_Bot_Logger::log( 'Search returned no statuses. Post may not be federated yet.', 'debug' );
                }
            } else {
                $search_body = wp_remote_retrieve_body( $search_response );
                Fifth_Social_Bot_Logger::log( 'Search API error (HTTP ' . $search_code . '): ' . substr( $search_body, 0, 200 ), 'warning' );
            }
        } else {
            Fifth_Social_Bot_Logger::log( 'Search API WP_Error: ' . $search_response->get_error_message(), 'warning' );
        }

        // If search didn't work, check if we're on the same instance (only then can we use ID from URL).
        // For cross-instance boost, we MUST find the post via search - ID from first instance won't work on second instance.
        if ( empty( $status_id ) ) {
            // Check if both URLs are from the same instance.
            $primary_instance = rtrim( trailingslashit( Fifth_Social_Bot_Core::get_instance_url() ), '/' );
            $secondary_instance_normalized = rtrim( $instance, '/' );
            
            // Only use ID from URL if instances are the same.
            if ( $primary_instance === $secondary_instance_normalized ) {
                $status_id = self::extract_status_id_from_url( $mastodon_url );
                if ( ! empty( $status_id ) ) {
                    Fifth_Social_Bot_Logger::log( 'Extracted status ID ' . $status_id . ' from URL (same instance)', 'debug' );
                }
            } else {
                Fifth_Social_Bot_Logger::log( 'Cannot use ID from URL for cross-instance boost. Primary: ' . $primary_instance . ', Secondary: ' . $secondary_instance_normalized, 'debug' );
            }
        }

        if ( empty( $status_id ) ) {
            // Post not found - likely not federated yet. Add to queue for retry later.
            $result['message'] = __( 'Could not find post on second instance. The post may not be federated yet. Added to boost queue for retry.', '5th-social-bot' );
            Fifth_Social_Bot_Logger::log(
                'Boost failed: Could not find post on ' . $instance . ' for URL: ' . $mastodon_url . '. Post may not be federated yet. Adding to queue for retry.',
                'info'
            );
            // Add to queue for retry later (federation can take a few minutes).
            self::enqueue_boost( $post_id );
            // Set next retry time (5 minutes from now).
            update_post_meta( $post_id, '_fifth_social_bot_boost_next_time', time() + 300 );
            return $result;
        }

        // Build boost URL: POST /api/v1/statuses/{id}/reblog
        $api_url = trailingslashit( $instance ) . 'api/v1/statuses/' . $status_id . '/reblog';
        
        Fifth_Social_Bot_Logger::log( 'Attempting to boost status ID ' . $status_id . ' on ' . $instance . ' via: ' . $api_url, 'debug' );

        // Make API request to boost.
        $response = wp_remote_post(
            $api_url,
            array(
                'timeout' => 15,
                'headers' => array(
                    'Authorization' => 'Bearer ' . $access_token,
                    'Content-Type'  => 'application/x-www-form-urlencoded',
                ),
            )
        );

        if ( is_wp_error( $response ) ) {
            $result['message'] = 'WP_Error: ' . $response->get_error_message();
            Fifth_Social_Bot_Logger::log(
                'Boost failed for post ID ' . $post_id . ' on ' . $instance . ': ' . $response->get_error_message(),
                'error'
            );
            // Add to queue for retry.
            self::enqueue_boost( $post_id );
            update_post_meta( $post_id, '_fifth_social_bot_boost_next_time', time() + 300 );
            return $result;
        }

        $code = wp_remote_retrieve_response_code( $response );
        Fifth_Social_Bot_Logger::log( 'Boost API response code: ' . $code, 'debug' );
        
        if ( $code < 200 || $code >= 300 ) {
            $response_body = wp_remote_retrieve_body( $response );
            $result['message'] = 'HTTP ' . $code . '. Response: ' . substr( $response_body, 0, 200 );
            
            // If 404, the status ID might be wrong or post not federated. Add to queue for retry.
            if ( 404 === $code ) {
                Fifth_Social_Bot_Logger::log(
                    'Boost failed with HTTP 404 for post ID ' . $post_id . ' on ' . $instance . '. Status ID ' . $status_id . ' not found. Post may not be federated yet. Adding to queue for retry.',
                    'info'
                );
                // Add to queue for retry later.
                self::enqueue_boost( $post_id );
                update_post_meta( $post_id, '_fifth_social_bot_boost_next_time', time() + 300 );
                $result['message'] = __( 'Post not found on second instance (HTTP 404). The post may not be federated yet. Added to boost queue for retry in 5 minutes.', '5th-social-bot' );
            } else {
                Fifth_Social_Bot_Logger::log(
                    'Boost failed for post ID ' . $post_id . ' on ' . $instance . '. HTTP ' . $code . ': ' . substr( $response_body, 0, 200 ),
                    'error'
                );
            }
            return $result;
        }

        // Extract boost URL from response.
        $body = wp_remote_retrieve_body( $response );
        $data = json_decode( $body, true );
        $boost_url = '';
        if ( is_array( $data ) && isset( $data['url'] ) ) {
            $boost_url = sanitize_url( $data['url'] );
        }

        $result['success'] = true;
        $result['url']    = $boost_url;
        $result['message'] = __( 'Post boosted successfully.', '5th-social-bot' );

        // Save boost URL to post meta so it can be displayed in admin.
        if ( ! empty( $boost_url ) ) {
            update_post_meta( $post_id, '_fifth_social_bot_boost_url', $boost_url );
        }

        // Update boost statistics.
        self::update_boost_stats();

        // Remove queued status if it was set.
        delete_post_meta( $post_id, '_fifth_social_bot_boost_queued' );
        delete_post_meta( $post_id, '_fifth_social_bot_boost_next_time' );

        Fifth_Social_Bot_Logger::log(
            'Post boosted on ' . $instance . ' for post ID ' . $post_id . '. Boost URL: ' . $boost_url,
            'info'
        );

        return $result;
    }

    /**
     * Extract status ID from Mastodon URL.
     *
     * @param string $url Mastodon post URL (e.g., https://social.5th.ro/@username/1234567890).
     * @return string Status ID or empty string.
     */
    protected static function extract_status_id_from_url( $url ) {
        if ( empty( $url ) ) {
            return '';
        }

        // Extract ID from URL pattern: https://instance/@username/1234567890
        // Or: https://instance/web/statuses/1234567890
        if ( preg_match( '#/(\d+)(?:/|$)#', $url, $matches ) ) {
            return $matches[1];
        }

        return '';
    }

    /**
     * Test connection to second instance.
     *
     * @param string $instance_url  Instance URL.
     * @param string $access_token   Access token.
     * @return array { ok: bool, message: string, username: string }
     */
    public static function test_boost_connection( $instance_url, $access_token ) {
        $result = array(
            'ok'       => false,
            'message'  => '',
            'username' => '',
        );

        Fifth_Social_Bot_Logger::log( 'test_boost_connection called with instance: ' . $instance_url . ' (token length: ' . strlen( $access_token ) . ')', 'debug' );

        if ( empty( $instance_url ) || empty( $access_token ) ) {
            $result['message'] = __( 'Missing instance URL or access token.', '5th-social-bot' );
            Fifth_Social_Bot_Logger::log( 'test_boost_connection failed: Missing instance URL or access token', 'error' );
            return $result;
        }

        // Validate and normalize instance URL.
        $url_validation = self::validate_instance_url( $instance_url );
        if ( ! $url_validation['valid'] ) {
            $result['message'] = $url_validation['message'];
            Fifth_Social_Bot_Logger::log( 'test_boost_connection failed: ' . $url_validation['message'], 'error' );
            return $result;
        }
        $instance = $url_validation['url'];

        // Clean access token.
        $access_token = trim( $access_token );
        $access_token = preg_replace( '/\s+/', '', $access_token );

        $url = trailingslashit( $instance ) . 'api/v1/accounts/verify_credentials';
        Fifth_Social_Bot_Logger::log( 'test_boost_connection: API URL: ' . $url, 'debug' );

        $response = wp_remote_get(
            $url,
            array(
                'timeout' => 15,
                'headers' => array(
                    'Authorization' => 'Bearer ' . $access_token,
                ),
            )
        );

        if ( is_wp_error( $response ) ) {
            $error_msg = $response->get_error_message();
            $result['message'] = 'WP error: ' . $error_msg;
            Fifth_Social_Bot_Logger::log( 'test_boost_connection failed with WP_Error: ' . $error_msg, 'error' );
            return $result;
        }

        $code = wp_remote_retrieve_response_code( $response );
        Fifth_Social_Bot_Logger::log( 'test_boost_connection: API response code: ' . $code, 'debug' );
        
        if ( $code < 200 || $code >= 300 ) {
            $response_body = wp_remote_retrieve_body( $response );
            $result['message'] = 'HTTP ' . $code . '. Response: ' . substr( $response_body, 0, 200 );
            Fifth_Social_Bot_Logger::log( 'test_boost_connection failed with HTTP ' . $code . '. Response: ' . substr( $response_body, 0, 200 ), 'error' );
            return $result;
        }

        $body = wp_remote_retrieve_body( $response );
        $data = json_decode( $body, true );
        if ( ! is_array( $data ) ) {
            $result['message'] = __( 'Invalid JSON response from Mastodon.', '5th-social-bot' );
            Fifth_Social_Bot_Logger::log( 'test_boost_connection failed: Invalid JSON response. Body: ' . substr( $body, 0, 200 ), 'error' );
            return $result;
        }

        $username = isset( $data['username'] ) ? trim( $data['username'] ) : '';
        $acct     = isset( $data['acct'] ) ? trim( $data['acct'] ) : '';

        // Extract username from acct if it contains @instance format.
        if ( ! empty( $acct ) && strpos( $acct, '@' ) !== false ) {
            $parts = explode( '@', $acct );
            if ( ! empty( $parts[0] ) ) {
                $username = $parts[0];
            }
        }

        $result['ok']       = true;
        $result['username'] = $username;
        $result['message'] = __( 'Connection successful.', '5th-social-bot' );

        Fifth_Social_Bot_Logger::log( 'test_boost_connection successful. Username: ' . ( ! empty( $username ) ? $username : 'N/A' ), 'info' );

        return $result;
    }
}

