<?php
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Analytics: collect and display Mastodon post statistics.
 */
class Fifth_Social_Bot_Analytics {

    /**
     * Get statistics for a Mastodon post URL.
     *
     * @param string $mastodon_url Mastodon post URL.
     * @return array Statistics (replies_count, reblogs_count, favourites_count, views_count).
     */
    public static function get_post_stats( $mastodon_url ) {
        if ( empty( $mastodon_url ) ) {
            return array(
                'replies_count'    => 0,
                'reblogs_count'    => 0,
                'favourites_count' => 0,
                'views_count'      => 0,
            );
        }

        // Check cache first (cache for 5 minutes).
        $cache_key = 'fsb_stats_' . md5( $mastodon_url );
        $cached = get_transient( $cache_key );
        if ( false !== $cached ) {
            return $cached;
        }

        // Extract instance and status ID from URL.
        $parsed = self::parse_mastodon_url( $mastodon_url );
        if ( empty( $parsed['instance'] ) || empty( $parsed['status_id'] ) ) {
            return array(
                'replies_count'    => 0,
                'reblogs_count'    => 0,
                'favourites_count' => 0,
                'views_count'      => 0,
            );
        }

        $settings = Fifth_Social_Bot_Core::get_settings();
        $access_token = ! empty( $settings['access_token'] ) ? trim( $settings['access_token'] ) : '';

        // If this is a boost URL, try to get stats from the boost instance.
        $boost_url = '';
        $boost_settings = array();
        if ( ! empty( $settings['boost_enabled'] ) && ! empty( $settings['boost_instance'] ) && ! empty( $settings['boost_access_token'] ) ) {
            $boost_settings = array(
                'instance'      => $settings['boost_instance'],
                'access_token' => $settings['boost_access_token'],
            );
        }

        // Try to get stats from primary instance first.
        $stats = self::fetch_stats_from_instance( $parsed['instance'], $parsed['status_id'], $access_token );
        
        // If not found and we have boost settings, try boost instance.
        if ( empty( $stats ) && ! empty( $boost_settings ) ) {
            $boost_parsed = self::parse_mastodon_url( $mastodon_url );
            if ( ! empty( $boost_parsed['status_id'] ) ) {
                $stats = self::fetch_stats_from_instance( $boost_settings['instance'], $boost_parsed['status_id'], $boost_settings['access_token'] );
            }
        }

        // Default stats if fetch failed.
        if ( empty( $stats ) || ! is_array( $stats ) ) {
            $stats = array(
                'replies_count'    => 0,
                'reblogs_count'    => 0,
                'favourites_count' => 0,
                'views_count'      => 0,
            );
        }

        // Cache for 5 minutes.
        set_transient( $cache_key, $stats, 300 );

        return $stats;
    }

    /**
     * Fetch statistics from a Mastodon instance.
     *
     * @param string $instance    Instance URL.
     * @param string $status_id   Status ID.
     * @param string $access_token Access token.
     * @return array|false Statistics or false on error.
     */
    protected static function fetch_stats_from_instance( $instance, $status_id, $access_token ) {
        if ( empty( $instance ) || empty( $status_id ) || empty( $access_token ) ) {
            return false;
        }

        $instance = rtrim( $instance, '/' );
        $api_url = $instance . '/api/v1/statuses/' . absint( $status_id );

        $response = wp_remote_get(
            $api_url,
            array(
                'timeout' => 10,
                'headers' => array(
                    'Authorization' => 'Bearer ' . $access_token,
                ),
            )
        );

        if ( is_wp_error( $response ) ) {
            return false;
        }

        $code = wp_remote_retrieve_response_code( $response );
        if ( $code < 200 || $code >= 300 ) {
            return false;
        }

        $body = wp_remote_retrieve_body( $response );
        $data = json_decode( $body, true );

        if ( ! is_array( $data ) ) {
            return false;
        }

        return array(
            'replies_count'    => isset( $data['replies_count'] ) ? absint( $data['replies_count'] ) : 0,
            'reblogs_count'    => isset( $data['reblogs_count'] ) ? absint( $data['reblogs_count'] ) : 0,
            'favourites_count' => isset( $data['favourites_count'] ) ? absint( $data['favourites_count'] ) : 0,
            'views_count'      => isset( $data['views_count'] ) ? absint( $data['views_count'] ) : 0, // May not be available on all instances.
        );
    }

    /**
     * Parse Mastodon URL to extract instance and status ID.
     *
     * @param string $url Mastodon post URL.
     * @return array Array with 'instance' and 'status_id' keys.
     */
    protected static function parse_mastodon_url( $url ) {
        $result = array(
            'instance'  => '',
            'status_id' => '',
        );

        if ( empty( $url ) ) {
            return $result;
        }

        // Pattern: https://instance/@username/1234567890 or https://instance/web/statuses/1234567890
        if ( preg_match( '#^https?://([^/]+)/.*?/(\d+)(?:/|$)#', $url, $matches ) ) {
            $result['instance']  = 'https://' . $matches[1];
            $result['status_id'] = $matches[2];
        }

        return $result;
    }

    /**
     * Get all posts with Mastodon URLs and their statistics.
     *
     * @param int $limit Maximum number of posts to return.
     * @return array Array of posts with statistics.
     */
    public static function get_all_posts_stats( $limit = 50 ) {
        // Check cache first (cache for 5 minutes).
        $cache_key = 'fsb_all_posts_stats_' . absint( $limit );
        $cached = get_transient( $cache_key );
        if ( false !== $cached ) {
            return $cached;
        }

        // Use WordPress Query API instead of direct database query.
        // Note: meta_query is necessary here to filter posts with Mastodon URLs.
        // Cache is used to minimize database queries.
        // phpcs:disable WordPress.DB.SlowDBQuery.slow_db_query_meta_query
        $query = new WP_Query(
            array(
                'post_type'      => array( 'post', 'page' ),
                'post_status'    => 'publish',
                'posts_per_page' => absint( $limit ),
                'orderby'        => 'date',
                'order'          => 'DESC',
                'fields'         => 'ids', // Only get IDs for better performance.
                'meta_query'     => array(
                    array(
                        'key'     => '_fifth_social_bot_mastodon_url',
                        'value'   => '',
                        'compare' => '!=',
                    ),
                ),
            )
        );
        // phpcs:enable WordPress.DB.SlowDBQuery.slow_db_query_meta_query

        $posts = array();
        $post_ids = $query->get_posts();
        if ( ! empty( $post_ids ) ) {
            foreach ( $post_ids as $post_id ) {
                $mastodon_url = get_post_meta( $post_id, '_fifth_social_bot_mastodon_url', true );
                if ( ! empty( $mastodon_url ) ) {
                    $post_obj = get_post( $post_id );
                    if ( $post_obj ) {
                        $posts[] = array(
                            'ID'            => $post_id,
                            'post_title'    => $post_obj->post_title,
                            'post_date'     => $post_obj->post_date,
                            'mastodon_url'  => $mastodon_url,
                        );
                    }
                }
            }
        }

        $results = array();
        foreach ( $posts as $post ) {
            $stats = self::get_post_stats( $post['mastodon_url'] );
            $boost_url = get_post_meta( $post['ID'], '_fifth_social_bot_boost_url', true );
            $boost_stats = array();
            if ( ! empty( $boost_url ) ) {
                $boost_stats = self::get_post_stats( $boost_url );
            }

            $results[] = array(
                'post_id'      => (int) $post['ID'],
                'post_title'   => $post['post_title'],
                'post_date'    => $post['post_date'],
                'mastodon_url' => $post['mastodon_url'],
                'boost_url'    => $boost_url,
                'stats'        => $stats,
                'boost_stats'  => $boost_stats,
            );
        }

        // Cache results for 5 minutes.
        set_transient( $cache_key, $results, 300 );

        return $results;
    }

    /**
     * Get aggregated statistics.
     *
     * @return array Aggregated stats.
     */
    public static function get_aggregated_stats() {
        $all_stats = self::get_all_posts_stats( 100 );
        
        $total_posts = count( $all_stats );
        $total_replies = 0;
        $total_reblogs = 0;
        $total_favourites = 0;
        $total_views = 0;
        $total_boost_replies = 0;
        $total_boost_reblogs = 0;
        $total_boost_favourites = 0;
        $total_boost_views = 0;
        $posts_with_boosts = 0;

        foreach ( $all_stats as $post_stats ) {
            $total_replies += $post_stats['stats']['replies_count'];
            $total_reblogs += $post_stats['stats']['reblogs_count'];
            $total_favourites += $post_stats['stats']['favourites_count'];
            $total_views += $post_stats['stats']['views_count'];

            if ( ! empty( $post_stats['boost_url'] ) ) {
                $posts_with_boosts++;
                $total_boost_replies += $post_stats['boost_stats']['replies_count'];
                $total_boost_reblogs += $post_stats['boost_stats']['reblogs_count'];
                $total_boost_favourites += $post_stats['boost_stats']['favourites_count'];
                $total_boost_views += $post_stats['boost_stats']['views_count'];
            }
        }

        return array(
            'total_posts'            => $total_posts,
            'total_replies'          => $total_replies,
            'total_reblogs'          => $total_reblogs,
            'total_favourites'       => $total_favourites,
            'total_views'            => $total_views,
            'posts_with_boosts'      => $posts_with_boosts,
            'total_boost_replies'    => $total_boost_replies,
            'total_boost_reblogs'    => $total_boost_reblogs,
            'total_boost_favourites' => $total_boost_favourites,
            'total_boost_views'       => $total_boost_views,
            'avg_replies'            => $total_posts > 0 ? round( $total_replies / $total_posts, 1 ) : 0,
            'avg_reblogs'            => $total_posts > 0 ? round( $total_reblogs / $total_posts, 1 ) : 0,
            'avg_favourites'         => $total_posts > 0 ? round( $total_favourites / $total_posts, 1 ) : 0,
            'avg_views'              => $total_posts > 0 ? round( $total_views / $total_posts, 1 ) : 0,
        );
    }
}

