<?php
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Admin settings pages and tabs.
 */
class Fifth_Social_Bot_Admin {

    public function __construct() {
        add_action( 'admin_menu', array( $this, 'add_menu' ) );
        add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_admin_styles' ) );
        // Per-post control meta box.
        add_action( 'add_meta_boxes', array( $this, 'add_per_post_meta_box' ) );
        add_action( 'save_post', array( $this, 'save_per_post_meta_box' ), 10, 2 );
    }

    /**
     * Enqueue admin styles for better UI.
     */
    public function enqueue_admin_styles() {
        // Add global CSS to preserve icon colors regardless of WordPress filters
        ?>
        <style id="fifth-social-bot-menu-icon-styles">
            /* Preserve original SVG colors for 5th Social Bot menu icon */
            #toplevel_page_fifth-social-bot .wp-menu-image svg,
            #toplevel_page_fifth-social-bot .wp-menu-image svg * {
                filter: none !important;
                -webkit-filter: none !important;
            }
            /* Ensure badge and text remain visible - force black badge and white text */
            #toplevel_page_fifth-social-bot .wp-menu-image svg rect {
                fill: #000000 !important;
            }
            #toplevel_page_fifth-social-bot .wp-menu-image svg text {
                fill: #ffffff !important;
                opacity: 1 !important;
            }
        </style>
        <?php
        
        $screen = get_current_screen();
        if ( $screen && ( 'toplevel_page_fifth-social-bot' === $screen->id || 'settings_page_fifth-social-bot' === $screen->id ) ) {
            ?>
            <style>
                /* Plugin Admin Styles */
                .fifth-social-bot-wrap {
                    max-width: 1200px;
                }
                
                /* Card-style sections */
                .fsb-card {
                    background: #fff;
                    border: 1px solid #c3c4c7;
                    border-radius: 4px;
                    box-shadow: 0 1px 1px rgba(0, 0, 0, 0.04);
                    padding: 20px;
                    margin: 20px 0;
                }
                
                .fsb-card h2 {
                    margin-top: 0;
                    padding-bottom: 10px;
                    border-bottom: 1px solid #e5e5e5;
                }
                
                .fsb-card h3 {
                    margin-top: 20px;
                    margin-bottom: 15px;
                    color: #1d2327;
                }
                
                /* Info boxes */
                .fsb-info-box {
                    background: #f0f6fc;
                    border-left: 4px solid #72aee6;
                    padding: 12px 15px;
                    margin: 15px 0;
                    border-radius: 2px;
                }
                
                .fsb-info-box.success {
                    background: #edfaef;
                    border-left-color: #00a32a;
                }
                
                .fsb-info-box.warning {
                    background: #fcf9e8;
                    border-left-color: #dba617;
                }
                
                .fsb-info-box.error {
                    background: #fcf0f1;
                    border-left-color: #d63638;
                }
                
                /* Status indicator */
                .fsb-status-indicator {
                    display: inline-block;
                    padding: 8px 12px;
                    border-radius: 4px;
                    margin: 10px 0;
                    font-weight: 500;
                }
                
                .fsb-status-indicator.connected {
                    background: #edfaef;
                    border-left: 4px solid #00a32a;
                    color: #1d2327;
                }
                
                .fsb-status-indicator.warning {
                    background: #fcf9e8;
                    border-left: 4px solid #dba617;
                    color: #1d2327;
                }
                
                .fsb-status-indicator.error {
                    background: #fcf0f1;
                    border-left: 4px solid #d63638;
                    color: #1d2327;
                }
                
                /* Tables */
                .fsb-card table.widefat {
                    margin-top: 15px;
                }
                
                /* Buttons group */
                .fsb-button-group {
                    display: flex;
                    gap: 10px;
                    flex-wrap: wrap;
                    margin: 15px 0;
                }
                
                /* Stats cards */
                .fsb-stats-grid {
                    display: grid;
                    grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
                    gap: 15px;
                    margin: 20px 0;
                }
                
                .fsb-stat-card {
                    background: #f6f7f7;
                    border: 1px solid #dcdcde;
                    border-radius: 4px;
                    padding: 15px;
                    text-align: center;
                }
                
                .fsb-stat-card .stat-value {
                    font-size: 28px;
                    font-weight: 600;
                    color: #2271b1;
                    margin: 10px 0;
                }
                
                .fsb-stat-card .stat-label {
                    color: #646970;
                    font-size: 13px;
                    text-transform: uppercase;
                    letter-spacing: 0.5px;
                }
                
                /* Form sections */
                .fsb-form-section {
                    background: #fff;
                    border: 1px solid #dcdcde;
                    border-radius: 4px;
                    padding: 20px;
                    margin: 20px 0;
                }
                
                .fsb-form-section h3 {
                    margin-top: 0;
                    padding-bottom: 10px;
                    border-bottom: 1px solid #e5e5e5;
                }
                
                /* Code blocks */
                .fsb-code-block {
                    background: #f6f7f7;
                    border: 1px solid #dcdcde;
                    border-radius: 4px;
                    padding: 15px;
                    margin: 15px 0;
                    font-family: 'Courier New', monospace;
                    font-size: 13px;
                    overflow-x: auto;
                }
                
                /* Responsive */
                @media (max-width: 782px) {
                    .fsb-stats-grid {
                        grid-template-columns: 1fr;
                    }
                    
                    .fsb-button-group {
                        flex-direction: column;
                    }
                    
                    .fsb-button-group .button {
                        width: 100%;
                    }
                }
            </style>
            <?php
        }
    }

    /**
     * Add per-post control meta box.
     */
    public function add_per_post_meta_box() {
        $post_types = array( 'post', 'page' );
        foreach ( $post_types as $post_type ) {
            add_meta_box(
                'fifth_social_bot_per_post_control',
                __( 'Mastodon Posting', '5th-social-bot' ),
                array( $this, 'render_per_post_meta_box' ),
                $post_type,
                'side',
                'default'
            );
        }
    }

    /**
     * Render per-post control meta box.
     *
     * @param WP_Post $post Post object.
     */
    public function render_per_post_meta_box( $post ) {
        wp_nonce_field( 'fifth_social_bot_per_post_control', 'fifth_social_bot_per_post_control_nonce' );
        
        $exclude_from_mastodon = get_post_meta( $post->ID, '_fifth_social_bot_exclude', true );
        $checked = ! empty( $exclude_from_mastodon ) ? 'checked="checked"' : '';
        
        ?>
        <p>
            <label>
                <input type="checkbox" name="fifth_social_bot_exclude" value="1" <?php echo esc_attr( $checked ); ?> />
                <?php echo esc_html__( 'Exclude from Mastodon auto-posting', '5th-social-bot' ); ?>
            </label>
        </p>
        <p class="description">
            <?php echo esc_html__( 'If checked, this post/page will not be automatically posted to Mastodon when published.', '5th-social-bot' ); ?>
        </p>
        <?php
    }

    /**
     * Save per-post control meta box.
     *
     * @param int     $post_id Post ID.
     * @param WP_Post $post    Post object.
     */
    public function save_per_post_meta_box( $post_id, $post ) {
        // Check nonce.
        if ( ! isset( $_POST['fifth_social_bot_per_post_control_nonce'] ) || 
             ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['fifth_social_bot_per_post_control_nonce'] ) ), 'fifth_social_bot_per_post_control' ) ) {
            return;
        }

        // Check autosave.
        if ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE ) {
            return;
        }

        // Check permissions.
        if ( ! current_user_can( 'edit_post', $post_id ) ) {
            return;
        }

        // Check post type.
        if ( ! in_array( $post->post_type, array( 'post', 'page' ), true ) ) {
            return;
        }

        // Save meta.
        if ( isset( $_POST['fifth_social_bot_exclude'] ) && '1' === $_POST['fifth_social_bot_exclude'] ) {
            update_post_meta( $post_id, '_fifth_social_bot_exclude', '1' );
        } else {
            delete_post_meta( $post_id, '_fifth_social_bot_exclude' );
        }
    }

    /**
     * Add settings page.
     */
    public function add_menu() {
        // Create SVG icon data URI (Mastodon icon with number 5)
        // Using pure white design optimized for WordPress admin menu filters
        // Mastodon icon: white filled shape, number 5: white text in dark circle badge for contrast
        // Simple "5th" text icon - clean design that works with WordPress admin filters
        // White text on transparent background, optimized for WordPress menu
        $icon_svg = 'data:image/svg+xml;base64,' . base64_encode( '<svg width="20" height="20" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg"><text x="12" y="16" font-size="14" font-weight="900" fill="#ffffff" text-anchor="middle" font-family="Arial, Helvetica, sans-serif">5th</text></svg>' );
        
        add_menu_page(
            __( '5th Social Bot (Mastodon All-In-One Wordpress Plugin)', '5th-social-bot' ),
            __( '5th Social Bot', '5th-social-bot' ),
            'manage_options',
            'fifth-social-bot',
            array( $this, 'render_settings_page' ),
            $icon_svg,
            30 // Position after Dashboard
        );
    }

    /**
     * Render main settings page (with tabs).
     */
    public function render_settings_page() {
        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }

        $settings            = Fifth_Social_Bot_Core::get_settings();
        $message             = '';
        $message_class       = 'updated';
        $connection_message  = '';
        $template_preview    = '';
        $boost_connection_message = '';
        $connection_status   = null; // Will be set if we check connection.

        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- read-only tab switch, no persistence.
        $current_tab = isset( $_GET['tab'] ) ? sanitize_key( wp_unslash( $_GET['tab'] ) ) : 'settings';
        if ( ! in_array( $current_tab, array( 'settings', 'bulk', 'shortcode', 'boost', 'analytics', 'log', 'support', 'terms' ), true ) ) {
            $current_tab = 'settings';
        }

        $settings_url = admin_url( 'admin.php?page=fifth-social-bot' );

        if ( isset( $_SERVER['REQUEST_METHOD'] ) && 'POST' === $_SERVER['REQUEST_METHOD'] && isset( $_POST['fifth_social_bot_action'] ) ) {
            check_admin_referer( 'fifth_social_bot_settings' );
            $action = sanitize_text_field( wp_unslash( $_POST['fifth_social_bot_action'] ) );

            if ( 'save_settings' === $action ) {
                $old_access_token = $settings['access_token'];
                // Clean access token - remove any whitespace/newlines.
                // Only update if the field is present in POST (to avoid resetting when saving from other tabs).
                if ( isset( $_POST['fsb_access_token'] ) ) {
                    $raw_token = sanitize_text_field( wp_unslash( $_POST['fsb_access_token'] ) );
                    $settings['access_token'] = preg_replace( '/\s+/', '', trim( $raw_token ) );
                } else {
                    // Keep existing token if not in POST (e.g., when saving from Auto-Boost tab).
                    $settings['access_token'] = $old_access_token;
                }
                // Only update these settings if they are present in POST (to avoid resetting when saving from other tabs).
                if ( isset( $_POST['fsb_visibility'] ) ) {
                    $settings['visibility'] = sanitize_text_field( wp_unslash( $_POST['fsb_visibility'] ) );
                }
                if ( isset( $_POST['fsb_language'] ) ) {
                    $settings['language'] = sanitize_text_field( wp_unslash( $_POST['fsb_language'] ) );
                }
                if ( isset( $_POST['fsb_post_posts'] ) ) {
                    $settings['post_posts'] = 1;
                } elseif ( isset( $_POST['fsb_template'] ) || isset( $_POST['fsb_visibility'] ) ) {
                    // Only reset if we're on the Settings tab (other Settings fields are present).
                    $settings['post_posts'] = 0;
                }
                if ( isset( $_POST['fsb_post_pages'] ) ) {
                    $settings['post_pages'] = 1;
                } elseif ( isset( $_POST['fsb_template'] ) || isset( $_POST['fsb_visibility'] ) ) {
                    // Only reset if we're on the Settings tab (other Settings fields are present).
                    $settings['post_pages'] = 0;
                }
                if ( isset( $_POST['fsb_template'] ) ) {
                    $settings['template'] = sanitize_textarea_field( wp_unslash( $_POST['fsb_template'] ) );
                }
                if ( isset( $_POST['fsb_readmore_texts'] ) ) {
                    $settings['readmore_texts'] = sanitize_textarea_field( wp_unslash( $_POST['fsb_readmore_texts'] ) );
                }
                // Username is read-only, keep existing value or it will be set automatically below.
                // Don't read from POST since the field is readonly.
                
                // Handle verification checkbox - preserve existing value if not in POST, or set from POST.
                if ( isset( $_POST['fsb_enable_verification'] ) ) {
                    $settings['enable_verification'] = 1;
                } elseif ( isset( $_POST['fsb_template'] ) || isset( $_POST['fsb_visibility'] ) ) {
                    // Only reset if we're on the Settings tab (other Settings fields are present).
                    // Keep existing value if checkbox not in POST and we're not on Settings tab.
                    if ( ! isset( $settings['enable_verification'] ) ) {
                        $settings['enable_verification'] = 0;
                    }
                }
                
                // Handle debug mode checkbox.
                if ( isset( $_POST['fsb_debug_mode'] ) ) {
                    $settings['debug_mode'] = 1;
                } elseif ( isset( $_POST['fsb_template'] ) || isset( $_POST['fsb_visibility'] ) ) {
                    // Only reset if we're on the Settings tab (other Settings fields are present).
                    $settings['debug_mode'] = 0;
                }

                // Handle custom instance checkbox and URL.
                if ( isset( $_POST['fsb_use_custom_instance'] ) ) {
                    $settings['use_custom_instance'] = 1;
                    // Validate and save instance URL.
                    if ( isset( $_POST['fsb_mastodon_instance'] ) ) {
                        $raw_instance = sanitize_text_field( wp_unslash( $_POST['fsb_mastodon_instance'] ) );
                        $raw_instance = trim( $raw_instance );
                        if ( ! empty( $raw_instance ) ) {
                            // Ensure it starts with http:// or https://
                            if ( ! preg_match( '#^https?://#', $raw_instance ) ) {
                                $raw_instance = 'https://' . $raw_instance;
                            }
                            // Remove trailing slash
                            $raw_instance = rtrim( $raw_instance, '/' );
                            // Validate URL format
                            if ( filter_var( $raw_instance, FILTER_VALIDATE_URL ) ) {
                                $settings['mastodon_instance'] = esc_url_raw( $raw_instance );
                            } else {
                                $settings['mastodon_instance'] = '';
                                add_settings_error(
                                    'fifth_social_bot_settings',
                                    'instance_url_invalid',
                                    __( 'Invalid instance URL format. Please enter a valid URL (e.g., https://mastodon.social).', '5th-social-bot' ),
                                    'error'
                                );
                            }
                        } else {
                            $settings['mastodon_instance'] = '';
                        }
                    }
                } else {
                    $settings['use_custom_instance'] = 0;
                    // Clear instance URL if checkbox is unchecked
                    if ( isset( $_POST['fsb_template'] ) || isset( $_POST['fsb_visibility'] ) ) {
                        $settings['mastodon_instance'] = '';
                    }
                }

                // Handle boost settings.
                if ( isset( $_POST['fsb_boost_enabled'] ) ) {
                    $settings['boost_enabled'] = 1;
                } else {
                    $settings['boost_enabled'] = 0;
                }
                
                $raw_boost_instance = isset( $_POST['fsb_boost_instance'] ) ? sanitize_text_field( wp_unslash( $_POST['fsb_boost_instance'] ) ) : '';
                
                // Validate boost instance URL if provided.
                if ( ! empty( $raw_boost_instance ) ) {
                    $url_validation = Fifth_Social_Bot_Boost::validate_instance_url( $raw_boost_instance );
                    if ( ! $url_validation['valid'] ) {
                        // Show error message but don't prevent saving (user might fix it later).
                        add_settings_error(
                            'fifth_social_bot_settings',
                            'boost_instance_invalid',
                            $url_validation['message'],
                            'error'
                        );
                        // Still save the value so user can see what they entered.
                        $settings['boost_instance'] = $raw_boost_instance;
                    } else {
                        $settings['boost_instance'] = $url_validation['url'];
                    }
                } else {
                    $settings['boost_instance'] = '';
                }
                
                $old_boost_token = isset( $settings['boost_access_token'] ) ? $settings['boost_access_token'] : '';
                $raw_boost_token = isset( $_POST['fsb_boost_access_token'] ) ? sanitize_text_field( wp_unslash( $_POST['fsb_boost_access_token'] ) ) : '';
                $settings['boost_access_token'] = preg_replace( '/\s+/', '', trim( $raw_boost_token ) );
                
                // Boost rate limit settings.
                $boost_max_per_day = isset( $_POST['fsb_boost_max_per_day'] ) ? (int) $_POST['fsb_boost_max_per_day'] : 10;
                $boost_min_interval = isset( $_POST['fsb_boost_min_interval'] ) ? (int) $_POST['fsb_boost_min_interval'] : 30;
                
                // Validate and clamp values.
                $settings['boost_max_per_day'] = max( 1, min( 10, $boost_max_per_day ) );
                $settings['boost_min_interval'] = max( 30, min( 120, $boost_min_interval ) ); // 30 min to 2h
                
                // Auto-fetch boost username if token is set and username is empty or token changed.
                $boost_token_changed = ( $old_boost_token !== $settings['boost_access_token'] );
                if ( ! empty( $settings['boost_access_token'] ) && ! empty( $settings['boost_instance'] ) && ( empty( $settings['boost_username'] ) || $boost_token_changed ) ) {
                    $boost_result = Fifth_Social_Bot_Boost::test_boost_connection( $settings['boost_instance'], $settings['boost_access_token'] );
                    if ( $boost_result['ok'] && ! empty( $boost_result['username'] ) ) {
                        $settings['boost_username'] = $boost_result['username'];
                    }
                }

                if ( ! in_array( $settings['visibility'], array( 'public', 'unlisted', 'private', 'direct' ), true ) ) {
                    $settings['visibility'] = 'public';
                }

                // Validate language code (ISO 639-1 or ISO 639-2: 2-3 lowercase letters).
                if ( ! empty( $settings['language'] ) && ! preg_match( '/^[a-z]{2,3}$/i', $settings['language'] ) ) {
                    $settings['language'] = 'ro';
                }
                if ( empty( $settings['language'] ) ) {
                    $settings['language'] = 'ro';
                }
                $settings['language'] = strtolower( $settings['language'] );

                // Auto-fetch username if access token is set and username is empty or token changed.
                $token_changed = ( $old_access_token !== $settings['access_token'] );
                $is_new_token = ( empty( $old_access_token ) && ! empty( $settings['access_token'] ) );
                
                if ( ! empty( $settings['access_token'] ) && ( empty( $settings['mastodon_username'] ) || $token_changed ) ) {
                    $instance_url = Fifth_Social_Bot_Core::get_instance_url( $settings );
                    $connection_result = Fifth_Social_Bot_Core::test_connection( $settings['access_token'], $instance_url );
                    if ( $connection_result['ok'] && ! empty( $connection_result['username'] ) ) {
                        $settings['mastodon_username'] = $connection_result['username'];
                        // Enable verification by default when token is set for the first time.
                        if ( $is_new_token ) {
                            $settings['enable_verification'] = 1;
                            $message = __( 'Settings have been saved. Username automatically retrieved from Mastodon. Profile verification enabled by default.', '5th-social-bot' );
                        } elseif ( $token_changed ) {
                            // Token changed but not first time - only update username, keep verification setting.
                            $message = __( 'Settings have been saved. Username automatically updated from Mastodon.', '5th-social-bot' );
                        } else {
                            $message = __( 'Settings have been saved. Username automatically retrieved from Mastodon.', '5th-social-bot' );
                        }
                    } else {
                        $message = __( 'Settings have been saved.', '5th-social-bot' );
                    }
                } else {
                    $message = __( 'Settings have been saved.', '5th-social-bot' );
                }

                Fifth_Social_Bot_Core::save_settings( $settings );
                $message_class = 'updated';

            } elseif ( 'check_connection' === $action ) {
                $instance_url = Fifth_Social_Bot_Core::get_instance_url( $settings );
                $result = Fifth_Social_Bot_Core::test_connection( $settings['access_token'], $instance_url );
                if ( $result['ok'] ) {
                    $connection_message = $result['message'];
                    $message            = __( 'Connection successful.', '5th-social-bot' );
                    $message_class      = 'updated';

                    // Auto-update username if empty.
                    if ( ! empty( $result['username'] ) && empty( $settings['mastodon_username'] ) ) {
                        $settings['mastodon_username'] = $result['username'];
                        // Enable verification by default when token is set.
                        if ( empty( $settings['enable_verification'] ) ) {
                            $settings['enable_verification'] = 1;
                        }
                        Fifth_Social_Bot_Core::save_settings( $settings );
                        $message .= ' ' . __( 'Username automatically retrieved and saved. Profile verification enabled by default.', '5th-social-bot' );
                    }

                    Fifth_Social_Bot_Logger::log( 'Connection test: ' . $result['message'] . ' (Instance: ' . $instance_url . ')', 'info' );
                } else {
                    $connection_message = $result['message'];
                    $message            = __( 'Connection test failed: ', '5th-social-bot' ) . $result['message'];
                    $message_class      = 'error';
                    Fifth_Social_Bot_Logger::log( 'Connection test error: ' . $result['message'] . ' (Instance: ' . $instance_url . ')', 'error' );
                }

            } elseif ( 'check_boost_connection' === $action ) {
                Fifth_Social_Bot_Logger::log( 'Boost connection test triggered', 'info' );
                
                $boost_instance = isset( $_POST['fsb_boost_instance'] ) ? sanitize_text_field( wp_unslash( $_POST['fsb_boost_instance'] ) ) : '';
                $boost_token = isset( $_POST['fsb_boost_access_token'] ) ? sanitize_text_field( wp_unslash( $_POST['fsb_boost_access_token'] ) ) : '';
                
                // Clean token.
                $boost_token = preg_replace( '/\s+/', '', trim( $boost_token ) );
                
                Fifth_Social_Bot_Logger::log( 'Testing boost connection to instance: ' . $boost_instance . ' (token length: ' . strlen( $boost_token ) . ')', 'debug' );
                
                if ( empty( $boost_instance ) || empty( $boost_token ) ) {
                    $boost_connection_message = __( 'Please provide both instance URL and access token.', '5th-social-bot' );
                    $message = __( 'Boost connection test failed: ', '5th-social-bot' ) . $boost_connection_message;
                    $message_class = 'error';
                    Fifth_Social_Bot_Logger::log( 'Boost connection test failed: Missing instance URL or access token', 'error' );
                } else {
                    $result = Fifth_Social_Bot_Boost::test_boost_connection( $boost_instance, $boost_token );
                    
                    if ( $result['ok'] ) {
                        $boost_connection_message = $result['message'];
                        if ( ! empty( $result['username'] ) ) {
                            $boost_connection_message .= ' ' . sprintf(
                                /* translators: %s: username */
                                __( 'Username: @%s', '5th-social-bot' ),
                                esc_html( $result['username'] )
                            );
                        }
                        $message = __( 'Boost connection test successful.', '5th-social-bot' );
                        $message_class = 'updated';
                        Fifth_Social_Bot_Logger::log( 'Boost connection test successful. Username: ' . ( ! empty( $result['username'] ) ? $result['username'] : 'N/A' ), 'info' );
                        
                        // Auto-update boost username in settings if provided.
                        if ( ! empty( $result['username'] ) ) {
                            $settings['boost_username'] = $result['username'];
                            // Validate and normalize instance URL before saving.
                            $url_validation = Fifth_Social_Bot_Boost::validate_instance_url( $boost_instance );
                            if ( $url_validation['valid'] ) {
                                $settings['boost_instance'] = $url_validation['url'];
                            } else {
                                // If URL is invalid, still save the raw value but log a warning.
                                $settings['boost_instance'] = $boost_instance;
                                Fifth_Social_Bot_Logger::log( 'Warning: Invalid instance URL saved: ' . $url_validation['message'], 'warning' );
                            }
                            $settings['boost_access_token'] = $boost_token;
                            Fifth_Social_Bot_Core::save_settings( $settings );
                            $message .= ' ' . __( 'Username automatically saved.', '5th-social-bot' );
                            Fifth_Social_Bot_Logger::log( 'Boost username automatically saved: ' . $result['username'], 'info' );
                        }
                    } else {
                        $boost_connection_message = $result['message'];
                        $message = __( 'Boost connection test failed: ', '5th-social-bot' ) . $result['message'];
                        $message_class = 'error';
                        Fifth_Social_Bot_Logger::log( 'Boost connection test failed: ' . $result['message'], 'error' );
                    }
                }

            } elseif ( 'reset_template' === $action ) {
                Fifth_Social_Bot_Logger::log( 'Template reset to default triggered', 'info' );
                $settings['template'] = Fifth_Social_Bot_Core::get_default_template();
                Fifth_Social_Bot_Core::save_settings( $settings );
                $message       = __( 'Template has been reset to default.', '5th-social-bot' );
                $message_class = 'updated';
                Fifth_Social_Bot_Logger::log( 'Template reset to default completed. New template: ' . $settings['template'], 'info' );

            } elseif ( 'preview_template' === $action ) {
                Fifth_Social_Bot_Logger::log( 'Template preview action triggered', 'info' );
                $settings['template']       = isset( $_POST['fsb_template'] ) ? sanitize_textarea_field( wp_unslash( $_POST['fsb_template'] ) ) : $settings['template'];
                $settings['readmore_texts'] = isset( $_POST['fsb_readmore_texts'] ) ? sanitize_textarea_field( wp_unslash( $_POST['fsb_readmore_texts'] ) ) : $settings['readmore_texts'];

                $args = array(
                    'post_type'      => array( 'post', 'page' ),
                    'post_status'    => 'publish',
                    'posts_per_page' => 1,
                    'orderby'        => 'date',
                    'order'          => 'DESC',
                    'no_found_rows'  => true,
                );

                $query = new WP_Query( $args );
                if ( $query->have_posts() ) {
                    $post             = $query->posts[0];
                    $template_preview = Fifth_Social_Bot_Core::compose_status( $post->ID, $settings );
                    $message          = __( 'Preview generated using the latest published post/page.', '5th-social-bot' );
                    $message_class    = 'updated';
                    Fifth_Social_Bot_Logger::log( 'Template preview generated successfully for post ID ' . $post->ID . '. Preview length: ' . strlen( $template_preview ) . ' characters', 'info' );
                } else {
                    $template_preview = '';
                    $message          = __( 'There are no published posts or pages yet for preview.', '5th-social-bot' );
                    $message_class    = 'error';
                    Fifth_Social_Bot_Logger::log( 'Template preview failed: No published posts or pages found', 'error' );
                }
                wp_reset_postdata();

            } elseif ( 'bulk_last' === $action ) {
                $count = isset( $_POST['fsb_bulk_count'] ) ? (int) $_POST['fsb_bulk_count'] : 0;
                if ( $count > 0 ) {
                    Fifth_Social_Bot_Logger::log( 'Bulk queue action triggered: adding ' . $count . ' latest posts to queue', 'info' );
                    $result = Fifth_Social_Bot_Queue::queue_latest( $count );
                    if ( ! empty( $result['error'] ) ) {
                        $message       = __( 'Bulk error: ', '5th-social-bot' ) . $result['error'];
                        $message_class = 'error';
                        Fifth_Social_Bot_Logger::log( 'Bulk queue action failed: ' . $result['error'], 'error' );
                    } else {
                        $message  = __( 'Bulk: posts added to Mastodon queue.', '5th-social-bot' );
                        $message .= ' ' . __( 'Added:', '5th-social-bot' ) . ' ' . (int) $result['added'] . ', ' . __( 'skipped:', '5th-social-bot' ) . ' ' . (int) $result['skipped'] . '.';
                        $message_class = 'updated';
                        Fifth_Social_Bot_Logger::log( 'Bulk queue action completed: ' . (int) $result['added'] . ' posts added, ' . (int) $result['skipped'] . ' skipped', 'info' );
                    }
                }

            } elseif ( 'post_single' === $action ) {
                $post_id = isset( $_POST['fsb_post_id'] ) ? (int) $_POST['fsb_post_id'] : 0;
                if ( $post_id > 0 ) {
                    Fifth_Social_Bot_Logger::log( 'Manual post action triggered for post ID ' . $post_id, 'info' );
                    $ok = Fifth_Social_Bot_Core::send_status( $post_id );
                    if ( $ok ) {
                        $message       = __( 'Post has been sent to Mastodon. Check Mastodon for details.', '5th-social-bot' ) . ' (ID ' . $post_id . ').';
                        $message_class = 'updated';
                        Fifth_Social_Bot_Logger::log( 'Manual post sent successfully for post ID ' . $post_id, 'info' );
                    } else {
                        $message       = __( 'Could not send post to Mastodon. Check the log.', '5th-social-bot' ) . ' (ID ' . $post_id . ').';
                        $message_class = 'error';
                        Fifth_Social_Bot_Logger::log( 'Manual post failed for post ID ' . $post_id, 'error' );
                    }
                }

            } elseif ( 'boost_single' === $action ) {
                $post_id = isset( $_POST['fsb_post_id'] ) ? (int) $_POST['fsb_post_id'] : 0;
                if ( $post_id > 0 ) {
                    Fifth_Social_Bot_Logger::log( 'Manual boost action triggered for post ID ' . $post_id, 'info' );
                    $mastodon_url = get_post_meta( $post_id, '_fifth_social_bot_mastodon_url', true );
                    if ( ! empty( $mastodon_url ) ) {
                        $boost_result = Fifth_Social_Bot_Boost::boost_post( $mastodon_url, $post_id );
                        if ( $boost_result['success'] && ! empty( $boost_result['url'] ) ) {
                            update_post_meta( $post_id, '_fifth_social_bot_boost_url', $boost_result['url'] );
                            delete_post_meta( $post_id, '_fifth_social_bot_boost_queued' );
                            delete_post_meta( $post_id, '_fifth_social_bot_boost_next_time' );
                            $message       = __( 'Post boosted successfully.', '5th-social-bot' ) . ' (ID ' . $post_id . ').';
                            $message_class = 'updated';
                            Fifth_Social_Bot_Logger::log( 'Manual boost successful for post ID ' . $post_id . '. Boost URL: ' . $boost_result['url'], 'info' );
                        } else {
                            // Rate limited - add to queue.
                            Fifth_Social_Bot_Boost::enqueue_boost( $post_id );
                            $message       = __( 'Boost rate limited. Added to boost queue.', '5th-social-bot' ) . ' (ID ' . $post_id . '). ' . ( ! empty( $boost_result['message'] ) ? $boost_result['message'] : '' );
                            $message_class = 'updated';
                            Fifth_Social_Bot_Logger::log( 'Manual boost rate limited for post ID ' . $post_id . '. Added to boost queue. Message: ' . ( ! empty( $boost_result['message'] ) ? $boost_result['message'] : 'Unknown' ), 'info' );
                        }
                    } else {
                        $message       = __( 'Post not found on Mastodon. Please post it first.', '5th-social-bot' ) . ' (ID ' . $post_id . ').';
                        $message_class = 'error';
                        Fifth_Social_Bot_Logger::log( 'Manual boost failed for post ID ' . $post_id . ': Post not found on Mastodon (no mastodon_url).', 'error' );
                    }
                }

            } elseif ( 'move_queue_up' === $action ) {
                $post_id = isset( $_POST['fsb_post_id'] ) ? (int) $_POST['fsb_post_id'] : 0;
                if ( $post_id > 0 ) {
                    $queue = Fifth_Social_Bot_Queue::get_queue();
                    $index = array_search( $post_id, $queue, true );
                    if ( false !== $index && $index > 0 ) {
                        // Swap with previous item.
                        $temp = $queue[ $index - 1 ];
                        $queue[ $index - 1 ] = $queue[ $index ];
                        $queue[ $index ] = $temp;
                        Fifth_Social_Bot_Queue::set_queue( $queue );
                        $message       = __( 'Post moved up in queue.', '5th-social-bot' );
                        $message_class = 'updated';
                        Fifth_Social_Bot_Logger::log( 'Post ID ' . $post_id . ' moved up in queue (from position ' . $index . ' to ' . ( $index - 1 ) . ')', 'info' );
                    }
                }

            } elseif ( 'move_queue_down' === $action ) {
                $post_id = isset( $_POST['fsb_post_id'] ) ? (int) $_POST['fsb_post_id'] : 0;
                if ( $post_id > 0 ) {
                    $queue = Fifth_Social_Bot_Queue::get_queue();
                    $index = array_search( $post_id, $queue, true );
                    if ( false !== $index && $index < count( $queue ) - 1 ) {
                        // Swap with next item.
                        $temp = $queue[ $index + 1 ];
                        $queue[ $index + 1 ] = $queue[ $index ];
                        $queue[ $index ] = $temp;
                        Fifth_Social_Bot_Queue::set_queue( $queue );
                        $message       = __( 'Post moved down in queue.', '5th-social-bot' );
                        $message_class = 'updated';
                        Fifth_Social_Bot_Logger::log( 'Post ID ' . $post_id . ' moved down in queue (from position ' . $index . ' to ' . ( $index + 1 ) . ')', 'info' );
                    }
                }

            } elseif ( 'move_boost_queue_up' === $action ) {
                $post_id = isset( $_POST['fsb_post_id'] ) ? (int) $_POST['fsb_post_id'] : 0;
                if ( $post_id > 0 ) {
                    $queue = Fifth_Social_Bot_Boost::get_boost_queue();
                    $index = array_search( $post_id, $queue, true );
                    if ( false !== $index && $index > 0 ) {
                        // Swap with previous item.
                        $temp = $queue[ $index - 1 ];
                        $queue[ $index - 1 ] = $queue[ $index ];
                        $queue[ $index ] = $temp;
                        Fifth_Social_Bot_Boost::set_boost_queue( $queue );
                        $message       = __( 'Post moved up in boost queue.', '5th-social-bot' );
                        $message_class = 'updated';
                        Fifth_Social_Bot_Logger::log( 'Post ID ' . $post_id . ' moved up in boost queue (from position ' . $index . ' to ' . ( $index - 1 ) . ')', 'info' );
                    }
                }

            } elseif ( 'move_boost_queue_down' === $action ) {
                $post_id = isset( $_POST['fsb_post_id'] ) ? (int) $_POST['fsb_post_id'] : 0;
                if ( $post_id > 0 ) {
                    $queue = Fifth_Social_Bot_Boost::get_boost_queue();
                    $index = array_search( $post_id, $queue, true );
                    if ( false !== $index && $index < count( $queue ) - 1 ) {
                        // Swap with next item.
                        $temp = $queue[ $index + 1 ];
                        $queue[ $index + 1 ] = $queue[ $index ];
                        $queue[ $index ] = $temp;
                        Fifth_Social_Bot_Boost::set_boost_queue( $queue );
                        $message       = __( 'Post moved down in boost queue.', '5th-social-bot' );
                        $message_class = 'updated';
                        Fifth_Social_Bot_Logger::log( 'Post ID ' . $post_id . ' moved down in boost queue (from position ' . $index . ' to ' . ( $index + 1 ) . ')', 'info' );
                    }
                }

            } elseif ( 'remove_boost_queue' === $action ) {
                $post_id = isset( $_POST['fsb_post_id'] ) ? (int) $_POST['fsb_post_id'] : 0;
                if ( $post_id > 0 ) {
                    Fifth_Social_Bot_Logger::log( 'Removing post ID ' . $post_id . ' from boost queue', 'info' );
                    $queue = Fifth_Social_Bot_Boost::get_boost_queue();
                    $index = array_search( $post_id, $queue, true );
                    if ( false !== $index ) {
                        // Remove from queue.
                        unset( $queue[ $index ] );
                        $queue = array_values( $queue ); // Re-index array.
                        Fifth_Social_Bot_Boost::set_boost_queue( $queue );
                        // Remove queued meta.
                        delete_post_meta( $post_id, '_fifth_social_bot_boost_queued' );
                        delete_post_meta( $post_id, '_fifth_social_bot_boost_next_time' );
                        $message       = __( 'Post removed from boost queue.', '5th-social-bot' ) . ' (ID ' . $post_id . ').';
                        $message_class = 'updated';
                        Fifth_Social_Bot_Logger::log( 'Post ID ' . $post_id . ' removed from boost queue', 'info' );
                    } else {
                        $message       = __( 'Post not found in boost queue.', '5th-social-bot' ) . ' (ID ' . $post_id . ').';
                        $message_class = 'error';
                        Fifth_Social_Bot_Logger::log( 'Failed to remove post ID ' . $post_id . ' from boost queue: not found', 'warning' );
                    }
                }

            } elseif ( 'clear_log' === $action ) {
                Fifth_Social_Bot_Logger::clear();
                $message       = __( 'Log has been cleared.', '5th-social-bot' );
                $message_class = 'updated';
            } elseif ( 'test_post' === $action ) {
                // Send a test post to Mastodon.
                if ( empty( $settings['access_token'] ) ) {
                    $message       = __( 'Cannot send test post: Access token is not set.', '5th-social-bot' );
                    $message_class = 'error';
                } else {
                    // Create a test status message.
                    $test_status = sprintf(
                        /* translators: %s: site name */
                        __( '🧪 Test post from %s - This is a test message from the 5th Social Bot (Mastodon All-In-One Wordpress Plugin). If you see this, your configuration is working correctly!', '5th-social-bot' ),
                        get_bloginfo( 'name' )
                    );
                    
                    // Send test post.
                    $test_result = Fifth_Social_Bot_Core::send_test_status( $test_status, $settings );
                    
                    if ( $test_result['success'] ) {
                        $message = sprintf(
                            /* translators: %s: Mastodon post URL */
                            __( 'Test post sent successfully! View it on Mastodon: %s', '5th-social-bot' ),
                            '<a href="' . esc_url( $test_result['url'] ) . '" target="_blank" rel="noopener noreferrer">' . esc_html( $test_result['url'] ) . '</a>'
                        );
                        $message_class = 'updated';
                        Fifth_Social_Bot_Logger::log( 'Test post sent successfully: ' . $test_result['url'], 'info' );
                    } else {
                        $message = __( 'Test post failed: ', '5th-social-bot' ) . ( isset( $test_result['message'] ) ? esc_html( $test_result['message'] ) : __( 'Unknown error', '5th-social-bot' ) );
                        $message_class = 'error';
                        Fifth_Social_Bot_Logger::log( 'Test post failed: ' . ( isset( $test_result['message'] ) ? $test_result['message'] : 'Unknown error' ), 'error' );
                    }
                }
            } elseif ( 'import_stats' === $action ) {
                // Import stats from Mastodon.
                if ( empty( $settings['access_token'] ) ) {
                    $message       = __( 'Cannot import stats: Access token is not set.', '5th-social-bot' );
                    $message_class = 'error';
                } else {
                    $instance_url = Fifth_Social_Bot_Core::get_instance_url( $settings );
                    $import_result = self::import_stats_from_mastodon( $settings, $instance_url );
                    if ( $import_result['success'] ) {
                        $message = sprintf(
                            /* translators: %d: number of posts imported */
                            __( 'Stats imported successfully! Found %d posts from Mastodon.', '5th-social-bot' ),
                            $import_result['count']
                        );
                        $message_class = 'updated';
                        Fifth_Social_Bot_Logger::log( 'Stats imported: ' . $import_result['count'] . ' posts found', 'info' );
                    } else {
                        $message = __( 'Import failed: ', '5th-social-bot' ) . ( isset( $import_result['message'] ) ? esc_html( $import_result['message'] ) : __( 'Unknown error', '5th-social-bot' ) );
                        $message_class = 'error';
                        Fifth_Social_Bot_Logger::log( 'Stats import failed: ' . ( isset( $import_result['message'] ) ? $import_result['message'] : 'Unknown error' ), 'error' );
                    }
                }
            }
        }
        ?>
        <div class="wrap">
            <h1><?php echo esc_html__( '5th Social Bot (Mastodon All-In-One Wordpress Plugin)', '5th-social-bot' ); ?></h1>
            
            <?php
            // Connection status indicator.
            if ( ! empty( $settings['access_token'] ) ) {
                // Quick check - don't do full API call on every page load, just check if token exists.
                // Full connection test is done via "Check connection" button.
                $has_token = ! empty( $settings['access_token'] );
                $has_username = ! empty( $settings['mastodon_username'] );
                $instance_url = Fifth_Social_Bot_Core::get_instance_url( $settings );
                $instance_domain = str_replace( array( 'https://', 'http://' ), '', $instance_url );
                
                if ( $has_token && $has_username ) {
                    $status_color = '#00a32a'; // Green.
                    $status_text = sprintf(
                        /* translators: %1$s: Mastodon username, %2$s: instance domain */
                        esc_html__( 'Connected as @%1$s@%2$s', '5th-social-bot' ),
                        esc_html( $settings['mastodon_username'] ),
                        esc_html( $instance_domain )
                    );
                    $status_icon = '✓';
                } elseif ( $has_token ) {
                    $status_color = '#dba617'; // Yellow/warning.
                    $status_text = esc_html__( 'Token set, but username not retrieved. Click "Check connection" to verify.', '5th-social-bot' );
                    $status_icon = '⚠';
                } else {
                    $status_color = '#d63638'; // Red.
                    $status_text = esc_html__( 'Not connected. Please set your Mastodon Access Token.', '5th-social-bot' );
                    $status_icon = '✗';
                }
                ?>
                <div class="fsb-status-indicator <?php echo esc_attr( $has_token && $has_username ? 'connected' : ( $has_token ? 'warning' : 'error' ) ); ?>">
                    <strong><?php echo esc_html( $status_icon ); ?> <?php echo esc_html( $status_text ); ?> <?php echo esc_html__( '(Main Account)', '5th-social-bot' ); ?></strong>
                </div>
                <?php
                
                // Show boost account status if configured.
                if ( ! empty( $settings['boost_enabled'] ) && ! empty( $settings['boost_instance'] ) && ! empty( $settings['boost_access_token'] ) ) {
                    $boost_has_username = ! empty( $settings['boost_username'] );
                    $boost_instance_domain = ! empty( $settings['boost_instance'] ) ? str_replace( array( 'https://', 'http://' ), '', rtrim( $settings['boost_instance'], '/' ) ) : '';
                    
                    if ( $boost_has_username ) {
                        $boost_status_class = 'connected';
                        if ( ! empty( $boost_instance_domain ) ) {
                            $boost_status_text = sprintf(
                                /* translators: %1$s: Boost account username, %2$s: instance domain */
                                esc_html__( 'Connected as @%1$s@%2$s', '5th-social-bot' ),
                                esc_html( $settings['boost_username'] ),
                                esc_html( $boost_instance_domain )
                            );
                        } else {
                            $boost_status_text = sprintf(
                                /* translators: %s: Boost account username */
                                esc_html__( 'Connected as @%s', '5th-social-bot' ),
                                esc_html( $settings['boost_username'] )
                            );
                        }
                        $boost_status_icon = '✓';
                    } else {
                        $boost_status_class = 'warning';
                        $boost_status_text = esc_html__( 'Token set, but username not retrieved. Click "Test connection with second instance" to verify.', '5th-social-bot' );
                        $boost_status_icon = '⚠';
                    }
                    ?>
                    <div class="fsb-status-indicator <?php echo esc_attr( $boost_status_class ); ?>" style="margin-top: 5px;">
                        <strong><?php echo esc_html( $boost_status_icon ); ?> <?php echo esc_html( $boost_status_text ); ?> <?php echo esc_html__( '(Boost Account)', '5th-social-bot' ); ?></strong>
                    </div>
                    <?php
                }
            } else {
                ?>
                <div class="fsb-status-indicator error">
                    <strong>✗ <?php echo esc_html__( 'Not connected. Please set your Mastodon Access Token in Settings.', '5th-social-bot' ); ?></strong>
                </div>
                <?php
            }
            ?>

            <h2 class="nav-tab-wrapper">
                <a href="<?php echo esc_url( $settings_url ); ?>" class="nav-tab <?php echo ( 'settings' === $current_tab ) ? 'nav-tab-active' : ''; ?>">
                    <?php echo esc_html( __( 'Settings', '5th-social-bot' ) ); ?>
                </a>
                <a href="<?php echo esc_url( add_query_arg( 'tab', 'log', $settings_url ) ); ?>" class="nav-tab <?php echo ( 'log' === $current_tab ) ? 'nav-tab-active' : ''; ?>">
                    <?php echo esc_html( __( 'Log & debug', '5th-social-bot' ) ); ?>
                </a>
                <a href="<?php echo esc_url( add_query_arg( 'tab', 'bulk', $settings_url ) ); ?>" class="nav-tab <?php echo ( 'bulk' === $current_tab ) ? 'nav-tab-active' : ''; ?>">
                    <?php echo esc_html( __( 'Bulk & queue', '5th-social-bot' ) ); ?>
                </a>
                <a href="<?php echo esc_url( add_query_arg( 'tab', 'boost', $settings_url ) ); ?>" class="nav-tab <?php echo ( 'boost' === $current_tab ) ? 'nav-tab-active' : ''; ?>">
                    <?php echo esc_html( __( 'Auto-Boost', '5th-social-bot' ) ); ?>
                </a>
                <a href="<?php echo esc_url( add_query_arg( 'tab', 'shortcode', $settings_url ) ); ?>" class="nav-tab <?php echo ( 'shortcode' === $current_tab ) ? 'nav-tab-active' : ''; ?>">
                    <?php echo esc_html( __( 'Shortcode', '5th-social-bot' ) ); ?>
                </a>
                <a href="<?php echo esc_url( add_query_arg( 'tab', 'analytics', $settings_url ) ); ?>" class="nav-tab <?php echo ( 'analytics' === $current_tab ) ? 'nav-tab-active' : ''; ?>">
                    <?php echo esc_html( __( 'Analytics', '5th-social-bot' ) ); ?>
                </a>
                <a href="<?php echo esc_url( add_query_arg( 'tab', 'terms', $settings_url ) ); ?>" class="nav-tab <?php echo ( 'terms' === $current_tab ) ? 'nav-tab-active' : ''; ?>">
                    <?php echo esc_html( __( 'Terms & Conditions', '5th-social-bot' ) ); ?>
                </a>
                <a href="<?php echo esc_url( add_query_arg( 'tab', 'support', $settings_url ) ); ?>" class="nav-tab <?php echo ( 'support' === $current_tab ) ? 'nav-tab-active' : ''; ?>">
                    <?php echo esc_html( __( 'Support & Contact', '5th-social-bot' ) ); ?>
                </a>
            </h2>

            <?php if ( ! empty( $message ) ) : ?>
                <div class="notice <?php echo esc_attr( $message_class ); ?> is-dismissible">
                    <p><?php echo esc_html( $message ); ?></p>
                </div>
            <?php endif; ?>

            <?php
            if ( 'settings' === $current_tab ) {
                $this->render_settings_tab( $settings, $connection_message, $template_preview );
            } elseif ( 'bulk' === $current_tab ) {
                $this->render_bulk_tab();
            } elseif ( 'shortcode' === $current_tab ) {
                $this->render_shortcode_tab( $settings );
            } elseif ( 'boost' === $current_tab ) {
                $this->render_boost_tab( $settings, $boost_connection_message );
            } elseif ( 'analytics' === $current_tab ) {
                $this->render_analytics_tab();
            } elseif ( 'log' === $current_tab ) {
                $this->render_log_tab();
            } elseif ( 'support' === $current_tab ) {
                Fifth_Social_Bot_Support::render_tab();
            } elseif ( 'terms' === $current_tab ) {
                Fifth_Social_Bot_Terms::render_tab();
            }
            ?>
        </div>
        <?php
    }

    /**
     * Settings tab.
     *
     * @param array  $settings           Settings.
     * @param string $connection_message Connection test message.
     * @param string $template_preview   Template preview text.
     */
    protected function render_settings_tab( $settings, $connection_message, $template_preview ) {
        ?>
        <form method="post" action="" id="fsb-settings-form">
            <?php wp_nonce_field( 'fifth_social_bot_settings' ); ?>
            <input type="hidden" name="fifth_social_bot_action" value="save_settings" id="fsb-action-field" />

            <div class="fsb-card">
                <h2><?php echo esc_html( __( 'Connection settings', '5th-social-bot' ) ); ?></h2>
                <table class="form-table" role="presentation">
                <tr>
                    <th scope="row">
                        <label for="fsb_access_token">
                            <?php echo esc_html( __( 'Mastodon Access Token', '5th-social-bot' ) ); ?>
                            <span class="dashicons dashicons-editor-help" style="margin-left:5px; color:#666; cursor:help; font-size:16px; vertical-align:middle;" title="<?php echo esc_attr__( 'Your Mastodon personal access token from social.5th.ro. This token allows the plugin to post on your behalf. Keep it secret!', '5th-social-bot' ); ?>"></span>
                        </label>
                    </th>
                    <td>
                        <div style="position: relative; display: inline-block; width: 100%; max-width: 500px;">
                            <input type="password" id="fsb_access_token" name="fsb_access_token" class="regular-text" value="<?php echo esc_attr( $settings['access_token'] ); ?>" autocomplete="off" style="padding-right: 50px; width: 100%;" />
                            <button type="button" id="fsb-toggle-token" class="button" style="position: absolute; right: 0; top: 0; height: 30px; padding: 0 10px; border-left: 1px solid #ddd; background: #f5f5f5; cursor: pointer;" title="<?php echo esc_attr__( 'Show/Hide token', '5th-social-bot' ); ?>">
                                <span id="fsb-toggle-token-text"><?php echo esc_html__( 'Show', '5th-social-bot' ); ?></span>
                            </button>
                        </div>
                        <p class="description">
                            <?php if ( empty( $settings['use_custom_instance'] ) ) : ?>
                                <span style="color: #dc3232; font-weight: bold;"><?php echo esc_html__( 'Important: This plugin works only with access tokens from', '5th-social-bot' ); ?> <a href="https://social.5th.ro" target="_blank" rel="noopener noreferrer">https://social.5th.ro</a>.</span>
                                <br />
                            <?php endif; ?>
                            <?php
                            if ( ! empty( $settings['use_custom_instance'] ) && ! empty( $settings['mastodon_instance'] ) ) {
                                $instance_display = esc_html( $settings['mastodon_instance'] );
                                printf(
                                    /* translators: %s: Mastodon instance URL */
                                    esc_html__( 'Personal access token generated in your %s account. Required permissions: write:statuses (to post statuses) and read:accounts (to retrieve your username).', '5th-social-bot' ),
                                    '<a href="' . esc_url( $settings['mastodon_instance'] ) . '" target="_blank" rel="noopener noreferrer">' . esc_html( $instance_display ) . '</a>'
                                );
                            } else {
                                printf(
                                    /* translators: %s: link to social.5th.ro */
                                    esc_html__( 'Personal access token generated in your %s account. Required permissions: write:statuses (to post statuses) and read:accounts (to retrieve your username).', '5th-social-bot' ),
                                    '<a href="https://social.5th.ro" target="_blank" rel="noopener noreferrer">social.5th.ro</a>'
                                );
                            }
                            ?>
                            <br />
                            <strong><?php echo esc_html__( 'Security:', '5th-social-bot' ); ?></strong>
                            <?php echo esc_html__( 'The token is hidden by default for security. Click "Show" to reveal it when needed.', '5th-social-bot' ); ?>
                        </p>
                    </td>
                </tr>

                <tr>
                    <th scope="row">
                        <label for="fsb_use_custom_instance">
                            <?php echo esc_html__( 'Instance Selection', '5th-social-bot' ); ?>
                        </label>
                    </th>
                    <td>
                        <label>
                            <input type="checkbox" name="fsb_use_custom_instance" id="fsb_use_custom_instance" value="1" <?php checked( isset( $settings['use_custom_instance'] ) ? $settings['use_custom_instance'] : 0, 1 ); ?> />
                            <?php echo esc_html__( "I'm not Romanian, I will use other instance", '5th-social-bot' ); ?>
                        </label>
                        <p class="description">
                            <?php echo esc_html__( 'If checked, you can use access tokens from any Mastodon instance, not just social.5th.ro.', '5th-social-bot' ); ?>
                        </p>
                    </td>
                </tr>

                <tr id="fsb_instance_url_row" style="<?php echo empty( $settings['use_custom_instance'] ) ? 'display:none;' : ''; ?>">
                    <th scope="row">
                        <label for="fsb_mastodon_instance">
                            <?php echo esc_html__( 'Mastodon Instance URL', '5th-social-bot' ); ?>
                        </label>
                    </th>
                    <td>
                        <input type="url" id="fsb_mastodon_instance" name="fsb_mastodon_instance" class="regular-text" value="<?php echo esc_attr( isset( $settings['mastodon_instance'] ) ? $settings['mastodon_instance'] : '' ); ?>" placeholder="https://mastodon.social" />
                        <p class="description">
                            <?php echo esc_html__( 'Enter your Mastodon instance URL (e.g., https://mastodon.social). The access token must be from this instance.', '5th-social-bot' ); ?>
                        </p>
                    </td>
                </tr>

                <tr>
                    <th scope="row">
                        <label for="fsb_visibility">
                            <?php echo esc_html( __( 'Status visibility', '5th-social-bot' ) ); ?>
                            <span class="dashicons dashicons-editor-help" style="margin-left:5px; color:#666; cursor:help; font-size:16px; vertical-align:middle;" title="<?php echo esc_attr__( 'Who can see your Mastodon posts: Public (everyone), Unlisted (not in public timeline), Private (followers only), Direct (mentioned users only).', '5th-social-bot' ); ?>"></span>
                        </label>
                    </th>
                    <td>
                        <select name="fsb_visibility" id="fsb_visibility">
                            <option value="public" <?php selected( $settings['visibility'], 'public' ); ?>>public</option>
                            <option value="unlisted" <?php selected( $settings['visibility'], 'unlisted' ); ?>>unlisted</option>
                            <option value="private" <?php selected( $settings['visibility'], 'private' ); ?>>private</option>
                            <option value="direct" <?php selected( $settings['visibility'], 'direct' ); ?>>direct</option>
                        </select>
                        <p class="description">
                            <?php echo esc_html( __( 'Visibility mode for the statuses sent to Mastodon.', '5th-social-bot' ) ); ?>
                        </p>
                    </td>
                </tr>

                <tr>
                    <th scope="row">
                        <label for="fsb_language">
                            <?php echo esc_html( __( 'Post language', '5th-social-bot' ) ); ?>
                            <span class="dashicons dashicons-editor-help" style="margin-left:5px; color:#666; cursor:help; font-size:16px; vertical-align:middle;" title="<?php echo esc_attr__( 'Language code for your posts (ISO 639-1 or ISO 639-2 format). Examples: ro (Romanian), en (English), fr (French).', '5th-social-bot' ); ?>"></span>
                        </label>
                    </th>
                    <td>
                        <input type="text" name="fsb_language" id="fsb_language" value="<?php echo esc_attr( isset( $settings['language'] ) ? $settings['language'] : 'ro' ); ?>" class="regular-text" maxlength="3" pattern="[a-z]{2,3}" style="text-transform:lowercase;" />
                        <p class="description">
                            <?php echo esc_html__( 'Language code for posts (ISO 639-1 or ISO 639-2 format, e.g., "ro" for Romanian, "en" for English). Default: ro (Romanian).', '5th-social-bot' ); ?>
                        </p>
                    </td>
                </tr>

                <tr>
                    <th scope="row"><label for="fsb_mastodon_username"><?php echo esc_html( __( 'Mastodon Username', '5th-social-bot' ) ); ?></label></th>
                    <td>
                        <?php
                        $username = isset( $settings['mastodon_username'] ) ? $settings['mastodon_username'] : '';
                        if ( ! empty( $username ) ) :
                            ?>
                            <input type="text" id="fsb_mastodon_username" class="regular-text" value="<?php echo esc_attr( '@' . ltrim( $username, '@' ) ); ?>" readonly="readonly" style="background-color:#f5f5f5; cursor:not-allowed;" />
                            <p class="description">
                                <?php echo esc_html__( 'Username is automatically retrieved from your Mastodon Access Token and cannot be edited manually.', '5th-social-bot' ); ?>
                            </p>
                        <?php else : ?>
                            <input type="text" id="fsb_mastodon_username" class="regular-text" value="" readonly="readonly" style="background-color:#f5f5f5; cursor:not-allowed;" placeholder="<?php echo esc_attr__( 'Will be set automatically when you save your Access Token', '5th-social-bot' ); ?>" />
                            <p class="description">
                                <?php echo esc_html__( 'Username will be automatically retrieved from your Mastodon Access Token when you save it.', '5th-social-bot' ); ?>
                            </p>
                        <?php endif; ?>
                    </td>
                </tr>

                <tr>
                    <th scope="row"><?php echo esc_html( __( 'Profile Verification', '5th-social-bot' ) ); ?></th>
                    <td>
                        <label>
                            <input type="checkbox" name="fsb_enable_verification" value="1" <?php checked( isset( $settings['enable_verification'] ) ? $settings['enable_verification'] : 0, 1 ); ?> />
                            <?php echo esc_html( __( 'Enable profile verification (adds rel="me" link and meta tag)', '5th-social-bot' ) ); ?>
                        </label>
                        <p class="description">
                            <?php echo esc_html__( 'Mastodon profile verification proves that you own both your website and your Mastodon account. When enabled, the plugin automatically adds verification code to your site header and footer.', '5th-social-bot' ); ?>
                        </p>
                        
                        <?php
                        $username = isset( $settings['mastodon_username'] ) ? $settings['mastodon_username'] : '';
                        $username = ltrim( $username, '@' );
                        $instance_url = Fifth_Social_Bot_Core::get_instance_url( $settings );
                        $profile_url = ! empty( $username ) ? trailingslashit( $instance_url ) . '@' . $username : '';
                        $enable_verification = isset( $settings['enable_verification'] ) ? $settings['enable_verification'] : 0;
                        ?>
                        
                        <?php if ( ! empty( $enable_verification ) && ! empty( $username ) ) : ?>
                            <div style="margin-top:15px; padding:15px; background:#f0f8ff; border-left:4px solid #0073aa; border-radius:2px;">
                                <p style="margin-top:0;">
                                    <strong><?php echo esc_html__( 'Status:', '5th-social-bot' ); ?></strong>
                                    <span style="color:green;">✓ <?php echo esc_html__( 'Verification enabled', '5th-social-bot' ); ?></span>
                                </p>
                                <p style="margin-bottom:10px;">
                                    <strong><?php echo esc_html__( 'Verification code added to your site:', '5th-social-bot' ); ?></strong>
                                </p>
                                <code style="display:block; padding:8px; background:#fff; border:1px solid #ddd; margin-bottom:8px;">
                                    &lt;meta name="fediverse:creator" content="@<?php 
                                    $instance_url = Fifth_Social_Bot_Core::get_instance_url( $settings );
                                    $instance_domain = str_replace( array( 'https://', 'http://' ), '', $instance_url );
                                    echo esc_html( $username . '@' . $instance_domain ); ?>"&gt;
                                </code>
                                <code style="display:block; padding:8px; background:#fff; border:1px solid #ddd;">
                                    &lt;a rel="me" href="<?php echo esc_url( $profile_url ); ?>"&gt;Mastodon&lt;/a&gt;
                                </code>
                                <p style="margin-top:15px; margin-bottom:0;">
                                    <strong><?php echo esc_html__( 'Important:', '5th-social-bot' ); ?></strong>
                                    <?php
                                    $site_url = home_url();
                                    $site_domain = wp_parse_url( $site_url, PHP_URL_HOST );
                                    $site_domain = preg_replace( '/^www\./', '', $site_domain );
                                    printf(
                                        /* translators: %1$s: site domain, %2$s: link to social.5th.ro */
                                        esc_html__( 'For verification to work, you must add your domain (%1$s) to the whitelist in your Mastodon account. Go to "Preferences → Public profile → Verification" on %2$s and add your domain to the whitelist.', '5th-social-bot' ),
                                        esc_html( $site_domain ),
                                        '<a href="https://social.5th.ro" target="_blank" rel="noopener noreferrer">social.5th.ro</a>'
                                    );
                                    ?>
                                </p>
                            </div>
                        <?php elseif ( ! empty( $enable_verification ) && empty( $username ) ) : ?>
                            <div style="margin-top:15px; padding:15px; background:#fff3cd; border-left:4px solid #ffc107; border-radius:2px;">
                                <p style="margin-top:0; margin-bottom:0;">
                                    <strong><?php echo esc_html__( 'Note:', '5th-social-bot' ); ?></strong>
                                    <?php echo esc_html__( 'Verification code will be added automatically once your Mastodon username is retrieved from your access token.', '5th-social-bot' ); ?>
                                </p>
                            </div>
                        <?php else : ?>
                            <div style="margin-top:15px; padding:15px; background:#f5f5f5; border-left:4px solid #999; border-radius:2px;">
                                <p style="margin-top:0; margin-bottom:0;">
                                    <strong><?php echo esc_html__( 'How it works:', '5th-social-bot' ); ?></strong>
                                    <?php echo esc_html__( 'Mastodon verifies account ownership by checking that your website contains a link back to your Mastodon profile with rel="me" attribute. When enabled, the plugin automatically adds the required verification code to your WordPress site header and footer.', '5th-social-bot' ); ?>
                                </p>
                            </div>
                        <?php endif; ?>
                    </td>
                </tr>
                
                <tr>
                    <th scope="row"><?php echo esc_html( __( 'Debug Mode', '5th-social-bot' ) ); ?></th>
                    <td>
                        <label>
                            <input type="checkbox" name="fsb_debug_mode" value="1" <?php checked( isset( $settings['debug_mode'] ) ? $settings['debug_mode'] : 0, 1 ); ?> />
                            <?php echo esc_html( __( 'Enable debug mode (logs additional information)', '5th-social-bot' ) ); ?>
                        </label>
                        <p class="description">
                            <?php echo esc_html__( 'When enabled, the plugin will log additional debug information to help troubleshoot issues. This does not affect plugin functionality.', '5th-social-bot' ); ?>
                        </p>
                    </td>
                </tr>
            </table>

            <h2><?php echo esc_html( __( 'Content types', '5th-social-bot' ) ); ?></h2>
            <table class="form-table" role="presentation">
                <tr>
                    <th scope="row"><?php echo esc_html( __( 'Auto-post for', '5th-social-bot' ) ); ?></th>
                    <td>
                        <label>
                            <input type="checkbox" name="fsb_post_posts" value="1" <?php checked( $settings['post_posts'], 1 ); ?> />
                            <?php echo esc_html( __( 'Posts (post)', '5th-social-bot' ) ); ?>
                        </label>
                        <br />
                        <label>
                            <input type="checkbox" name="fsb_post_pages" value="1" <?php checked( $settings['post_pages'], 1 ); ?> />
                            <?php echo esc_html( __( 'Pages (page)', '5th-social-bot' ) ); ?>
                        </label>
                    </td>
                </tr>
            </table>

            <h2><?php echo esc_html( __( 'Mastodon message customization', '5th-social-bot' ) ); ?></h2>
            <table class="form-table" role="presentation">
                <tr>
                    <th scope="row">
                        <label for="fsb_template">
                            <?php echo esc_html( __( 'Mastodon post template', '5th-social-bot' ) ); ?>
                            <span class="dashicons dashicons-editor-help" style="margin-left:5px; color:#666; cursor:help; font-size:16px; vertical-align:middle;" title="<?php echo esc_attr__( 'Template for Mastodon posts. Use placeholders: {text} for excerpt, {tags} for hashtags, {readmore_text} for call-to-action, {url} for post link. Use \\n for newlines.', '5th-social-bot' ); ?>"></span>
                        </label>
                    </th>
                    <td>
                        <textarea id="fsb_template" name="fsb_template" class="large-text code" rows="5"><?php echo esc_textarea( $settings['template'] ); ?></textarea>
                        <p style="margin-top:8px;">
                            <button type="button" id="fsb-reset-template" class="button">
                                <?php echo esc_html( __( 'Reset to Default', '5th-social-bot' ) ); ?>
                            </button>
                            <span class="description" style="margin-left:10px;">
                                <?php echo esc_html( __( 'Default template:', '5th-social-bot' ) ); ?> 
                                <code><?php echo esc_html( Fifth_Social_Bot_Core::get_default_template() ); ?></code>
                            </span>
                        </p>
                        <p class="description">
                            <?php echo esc_html( __( 'You can use placeholders: {text}, {tags}, {readmore_text}, {url}.', '5th-social-bot' ) ); ?>
                            <br />
                            <strong><?php echo esc_html__( 'Placeholders:', '5th-social-bot' ); ?></strong>
                            <br />
                            • <code>{text}</code> - <?php echo esc_html__( 'Excerpt or preview of your post/page content (automatically trimmed if too long)', '5th-social-bot' ); ?>
                            <br />
                            • <code>{tags}</code> - <?php echo esc_html__( 'WordPress tags converted to Mastodon hashtags (e.g., #WordPress #Blog)', '5th-social-bot' ); ?>
                            <br />
                            • <code>{readmore_text}</code> - <?php echo esc_html__( 'Random text from "Texts for {readmore_text}" field (e.g., "Read more at:")', '5th-social-bot' ); ?>
                            <br />
                            • <code>{url}</code> - <?php echo esc_html__( 'Permalink to your post or page', '5th-social-bot' ); ?>
                            <br /><br />
                            <?php echo esc_html( __( 'Default template: {text}\\n{tags}\\n{url} (each on a new line)', '5th-social-bot' ) ); ?>
                            <br />
                            <?php echo esc_html( __( 'You can use newlines by pressing Enter or writing \\n in the template (they will be converted to real newlines).', '5th-social-bot' ) ); ?>
                            <br />
                            <?php echo esc_html__( 'Mastodon has a 500-character limit. The plugin automatically trims the message if it exceeds this limit, prioritizing {url} and {tags}.', '5th-social-bot' ); ?>
                        </p>

                        <p class="description">
                            <?php echo esc_html( __( 'Quick emoji list (click to insert into the template):', '5th-social-bot' ) ); ?>
                        </p>
                        <div id="fsb-emoji-container">
                            <?php
                            // Complete Unicode emoji list (most common and useful emojis supported by Mastodon).
                            $emojis = array(
                                // Smileys & People
                                '😀', '😃', '😄', '😁', '😆', '😅', '😂', '🤣', '😊', '😇', '🙂', '🙃', '😉', '😌', '😍', '🥰', '😘', '😗', '😙', '😚', '😋', '😛', '😝', '😜', '🤪', '🤨', '🧐', '🤓', '😎', '🤩', '🥳', '😏', '😒', '😞', '😔', '😟', '😕', '🙁', '☹️', '😣', '😖', '😫', '😩', '🥺', '😢', '😭', '😤', '😠', '😡', '🤬', '🤯', '😳', '🥵', '🥶', '😱', '😨', '😰', '😥', '😓', '🤗', '🤔', '🤭', '🤫', '🤥', '😶', '😐', '😑', '😬', '🙄', '😯', '😦', '😧', '😮', '😲', '🥱', '😴', '🤤', '😪', '😵', '🤐', '🥴', '🤢', '🤮', '🤧', '😷', '🤒', '🤕', '🤑', '🤠', '😈', '👿', '👹', '👺', '🤡', '💩', '👻', '💀', '☠️', '👽', '👾', '🤖', '🎃', '😺', '😸', '😹', '😻', '😼', '😽', '🙀', '😿', '😾',
                                // Gestures & Body Parts
                                '👋', '🤚', '🖐️', '✋', '🖖', '👌', '🤏', '✌️', '🤞', '🤟', '🤘', '🤙', '👈', '👉', '👆', '🖕', '👇', '☝️', '👍', '👎', '✊', '👊', '🤛', '🤜', '👏', '🙌', '👐', '🤲', '🤝', '🙏', '✍️', '💪', '🦾', '🦿', '🦵', '🦶', '👂', '🦻', '👃', '🧠', '🦷', '🦴', '👀', '👁️', '👅', '👄',
                                // People
                                '👶', '🧒', '👦', '👧', '🧑', '👱', '👨', '🧔', '👩', '🧓', '👴', '👵', '🙍', '🙎', '🙅', '🙆', '💁', '🙋', '🧏', '🙇', '🤦', '🤷', '👮', '🕵️', '💂', '🥷', '👷', '🤴', '👸', '👳', '👲', '🧕', '🤵', '👰', '🤰', '🤱', '👼', '🎅', '🤶', '🦸', '🦹', '🧙', '🧚', '🧛', '🧜', '🧝', '🧞', '🧟', '💆', '💇', '🚶', '🧍', '🧎', '🏃', '💃', '🕺', '🕴️', '👯', '🧘', '🧗', '🤺', '🏇', '⛷️', '🏂', '🏌️', '🏄', '🚣', '🏊', '⛹️', '🏋️', '🚴', '🚵', '🤸', '🤼', '🤽', '🤾', '🤹', '🧗', '🧘', '🛀', '🛌', '👭', '👫', '👬', '💏', '💑', '👪', '👨‍👩‍👧', '👨‍👩‍👧‍👦', '👨‍👩‍👦‍👦', '👨‍👩‍👧‍👧', '👩‍👩‍👦', '👩‍👩‍👧', '👩‍👩‍👧‍👦', '👩‍👩‍👦‍👦', '👩‍👩‍👧‍👧', '👨‍👨‍👦', '👨‍👨‍👧', '👨‍👨‍👧‍👦', '👨‍👨‍👦‍👦', '👨‍👨‍👧‍👧', '👩‍👦', '👩‍👧', '👩‍👧‍👦', '👩‍👦‍👦', '👩‍👧‍👧', '👨‍👦', '👨‍👧', '👨‍👧‍👦', '👨‍👦‍👦', '👨‍👧‍👧',
                                // Animals & Nature
                                '🐶', '🐱', '🐭', '🐹', '🐰', '🦊', '🐻', '🐼', '🐨', '🐯', '🦁', '🐮', '🐷', '🐽', '🐸', '🐵', '🙈', '🙉', '🙊', '🐒', '🐔', '🐧', '🐦', '🐤', '🐣', '🐥', '🦆', '🦅', '🦉', '🦇', '🐺', '🐗', '🐴', '🦄', '🐝', '🐛', '🦋', '🐌', '🐞', '🐜', '🦟', '🦗', '🕷️', '🕸️', '🦂', '🐢', '🐍', '🦎', '🦖', '🦕', '🐙', '🦑', '🦐', '🦞', '🦀', '🐡', '🐠', '🐟', '🐬', '🐳', '🐋', '🦈', '🐊', '🐅', '🐆', '🦓', '🦍', '🦧', '🐘', '🦛', '🦏', '🐪', '🐫', '🦒', '🦘', '🦬', '🐃', '🐂', '🐄', '🐎', '🐖', '🐏', '🐑', '🦙', '🐐', '🦌', '🐕', '🐩', '🦮', '🐕‍🦺', '🐈', '🐓', '🦃', '🦤', '🦚', '🦜', '🦢', '🦩', '🕊️', '🐇', '🦝', '🦨', '🦡', '🦫', '🦦', '🦥', '🐁', '🐀', '🐿️', '🦔', '🌲', '🌳', '🌴', '🌵', '🌶️', '🌷', '🌺', '🌻', '🌼', '🌽', '🌾', '🌿', '☘️', '🍀', '🍁', '🍂', '🍃', '🍄', '🌰', '🌱', '🌿', '🍃', '🍂', '🍁', '🍄', '🌾', '🌽', '🌼', '🌻', '🌺', '🌷', '🌶️', '🌵', '🌴', '🌳', '🌲', '🌍', '🌎', '🌏', '🌕', '🌖', '🌗', '🌘', '🌑', '🌒', '🌓', '🌔', '🌙', '🌚', '🌛', '🌜', '🌝', '🌞', '⭐', '🌟', '💫', '✨', '☄️', '💥', '🔥', '🌈', '☀️', '⛅', '☁️', '⛈️', '🌤️', '🌦️', '🌧️', '⛈️', '🌩️', '🌨️', '❄️', '☃️', '⛄', '🌬️', '💨', '💧', '💦', '☔', '☂️', '🌊', '🌫️',
                                // Food & Drink
                                '🍏', '🍎', '🍐', '🍊', '🍋', '🍌', '🍉', '🍇', '🍓', '🍈', '🍒', '🍑', '🥭', '🍍', '🥥', '🥝', '🍅', '🍆', '🥑', '🥦', '🥬', '🥒', '🌶️', '🌽', '🥕', '🥔', '🍠', '🥐', '🥯', '🍞', '🥖', '🥨', '🧀', '🥚', '🍳', '🥞', '🥓', '🥩', '🍗', '🍖', '🦴', '🌭', '🍔', '🍟', '🍕', '🥪', '🥙', '🌮', '🌯', '🥗', '🥘', '🥫', '🍝', '🍜', '🍲', '🍛', '🍣', '🍱', '🥟', '🦪', '🍤', '🍙', '🍚', '🍘', '🍥', '🥠', '🥮', '🍢', '🍡', '🍧', '🍨', '🍦', '🥧', '🍰', '🎂', '🍮', '🍭', '🍬', '🍫', '🍿', '🍩', '🍪', '🌰', '🥜', '🍯', '🥛', '🍼', '☕', '🍵', '🥤', '🍶', '🍺', '🍻', '🥂', '🍷', '🥃', '🍸', '🍹', '🧃', '🧉', '🧊', '🥄', '🍴', '🍽️', '🥣', '🥡', '🥢',
                                // Activity & Sports
                                '⚽', '🏀', '🏈', '⚾', '🥎', '🎾', '🏐', '🏉', '🥏', '🎱', '🏓', '🏸', '🥅', '🏒', '🏑', '🏏', '🥍', '🏹', '🎣', '🥊', '🥋', '🎽', '🛹', '🛷', '⛸️', '🥌', '🎿', '⛷️', '🏂', '🏋️', '🤼', '🤸', '🤺', '🤾', '🏌️', '🏇', '🧘', '🏄', '🏊', '🚣', '🧗', '🚵', '🚴', '🏃', '🏃‍♀️', '🏃‍♂️', '🚶', '🚶‍♀️', '🚶‍♂️', '💃', '🕺', '🕴️', '👯', '🎪', '🎭', '🩰', '🎨', '🎬', '🎤', '🎧', '🎼', '🎹', '🥁', '🎷', '🎺', '🎸', '🪕', '🎻', '🎲', '♟️', '🎯', '🎳', '🎮', '🎰', '🧩',
                                // Travel & Places
                                '🚗', '🚕', '🚙', '🚌', '🚎', '🏎️', '🚓', '🚑', '🚒', '🚐', '🚚', '🚛', '🚜', '🛴', '🚲', '🛵', '🏍️', '🛺', '🚨', '🚔', '🚍', '🚘', '🚖', '🚡', '🚠', '🚟', '🚃', '🚋', '🚞', '🚝', '🚄', '🚅', '🚈', '🚂', '🚆', '🚇', '🚊', '🚉', '✈️', '🛫', '🛬', '🛩️', '💺', '🚁', '🚟', '🚠', '🚡', '🛰️', '🚀', '🛸', '🛶', '⛵', '🚤', '🛥️', '🛳️', '⛴️', '🚢', '⚓', '⛽', '🚧', '🚦', '🚥', '🗺️', '🗿', '🗽', '🗼', '🏰', '🏯', '🏟️', '🎡', '🎢', '🎠', '⛲', '⛱️', '🏖️', '🏝️', '🏜️', '🌋', '⛰️', '🏔️', '🗻', '🏕️', '⛺', '🏠', '🏡', '🏘️', '🏚️', '🏗️', '🏭', '🏢', '🏬', '🏣', '🏤', '🏥', '🏦', '🏨', '🏪', '🏫', '🏩', '💒', '🏛️', '⛪', '🕌', '🕍', '🕋', '⛩️', '🛤️', '🛣️', '🗾', '🎑', '🏞️', '🌅', '🌄', '🌠', '🎇', '🎆', '🌇', '🌆', '🏙️', '🌃', '🌌', '🌉', '🌁',
                                // Objects
                                '⌚', '📱', '📲', '💻', '⌨️', '🖥️', '🖨️', '🖱️', '🖲️', '🕹️', '🗜️', '💾', '💿', '📀', '📼', '📷', '📸', '📹', '🎥', '📽️', '🎞️', '📞', '☎️', '📟', '📠', '📺', '📻', '🎙️', '🎚️', '🎛️', '⏱️', '⏲️', '⏰', '🕰️', '⌛', '⏳', '📡', '🔋', '🔌', '💡', '🔦', '🕯️', '🧯', '🛢️', '💸', '💵', '💴', '💶', '💷', '💰', '💳', '💎', '⚖️', '🛠️', '🔧', '🔨', '⚒️', '🛠️', '⛏️', '🔩', '⚙️', '🧰', '🧲', '🧪', '🧫', '🧬', '🔬', '🔭', '📡', '💉', '💊', '🩹', '🩺', '🚪', '🛏️', '🛋️', '🚽', '🚿', '🛁', '🧴', '🧷', '🧹', '🧺', '🧻', '🧼', '🧽', '🧯', '🛒', '🚬', '⚰️', '⚱️', '🗿', '🏧', '🚮', '🚰', '♿', '🚹', '🚺', '🚻', '🚼', '🚾', '🛂', '🛃', '🛄', '🛅', '⚠️', '🚸', '⛔', '🚫', '🚳', '🚭', '🚯', '🚱', '🚷', '📵', '🔞', '☢️', '☣️', '⬆️', '↗️', '➡️', '↘️', '⬇️', '↙️', '⬅️', '↖️', '↕️', '↔️', '↩️', '↪️', '⤴️', '⤵️', '🔃', '🔄', '🔙', '🔚', '🔛', '🔜', '🔝', '🛐', '⚛️', '🕉️', '✡️', '☸️', '☯️', '✝️', '☦️', '☪️', '☮️', '🕎', '🔯', '♈', '♉', '♊', '♋', '♌', '♍', '♎', '♏', '♐', '♑', '♒', '♓', '⛎', '🔀', '🔁', '🔂', '▶️', '⏩', '⏭️', '⏯️', '◀️', '⏪', '⏮️', '🔼', '⏫', '🔽', '⏬', '⏸️', '⏹️', '⏺️', '⏏️', '🎦', '🔅', '🔆', '📶', '📳', '📴', '♠️', '♥️', '♦️', '♣️', '🃏', '🀄', '🎴', '🎭', '🖼️', '🎨', '🧩', '♟️', '🧸', '🪀', '🪁', '🪂', '🎯', '🎲', '🎰', '🎮', '🕹️', '🎳', '🎸', '🎺', '🎷', '🥁', '🎹', '🎤', '🎧', '🎼', '🎵', '🎶', '🎙️', '🎚️', '🎛️', '📻', '📺', '📷', '📸', '📹', '📽️', '🎞️', '🎬', '📱', '💻', '⌨️', '🖥️', '🖨️', '🖱️', '🖲️', '🕹️', '🗜️', '💾', '💿', '📀', '📼', '📷', '📸', '📹', '🎥', '📽️', '🎞️', '📞', '☎️', '📟', '📠', '📺', '📻', '🎙️', '🎚️', '🎛️', '⏱️', '⏲️', '⏰', '🕰️', '⌛', '⏳', '📡', '🔋', '🔌', '💡', '🔦', '🕯️', '🧯', '🛢️', '💸', '💵', '💴', '💶', '💷', '💰', '💳', '💎', '⚖️', '🛠️', '🔧', '🔨', '⚒️', '🛠️', '⛏️', '🔩', '⚙️', '🧰', '🧲', '🧪', '🧫', '🧬', '🔬', '🔭', '📡', '💉', '💊', '🩹', '🩺', '🚪', '🛏️', '🛋️', '🚽', '🚿', '🛁', '🧴', '🧷', '🧹', '🧺', '🧻', '🧼', '🧽', '🧯', '🛒', '🚬', '⚰️', '⚱️', '🗿', '🏧', '🚮', '🚰', '♿', '🚹', '🚺', '🚻', '🚼', '🚾', '🛂', '🛃', '🛄', '🛅', '⚠️', '🚸', '⛔', '🚫', '🚳', '🚭', '🚯', '🚱', '🚷', '📵', '🔞', '☢️', '☣️',
                                // Symbols
                                '❤️', '🧡', '💛', '💚', '💙', '💜', '🖤', '🤍', '🤎', '💔', '❣️', '💕', '💞', '💓', '💗', '💖', '💘', '💝', '💟', '☮️', '✝️', '☪️', '🕉️', '☸️', '✡️', '🔯', '🕎', '☯️', '☦️', '🛐', '⛎', '♈', '♉', '♊', '♋', '♌', '♍', '♎', '♏', '♐', '♑', '♒', '♓', '🆔', '⚛️', '🉑', '☢️', '☣️', '📴', '📳', '🈶', '🈚', '🈸', '🈺', '🈷️', '✴️', '🆚', '💮', '🉐', '㊙️', '㊗️', '🈴', '🈵', '🈹', '🈲', '🅰️', '🅱️', '🆎', '🆑', '🅾️', '🆘', '❌', '⭕', '🛑', '⛔', '📛', '🚫', '💯', '💢', '♨️', '🚷', '🚯', '🚳', '🚱', '🔞', '📵', '🚭', '❗', '❓', '❕', '❔', '‼️', '⁉️', '🔅', '🔆', '〽️', '⚠️', '🚸', '🔱', '⚜️', '🔰', '♻️', '✅', '🈯', '💹', '❇️', '✳️', '❎', '🌐', '💠', 'Ⓜ️', '🌀', '💤', '🏧', '🚾', '♿', '🅿️', '🈳', '🈂️', '🛂', '🛃', '🛄', '🛅', '🚹', '🚺', '🚼', '🚻', '🚮', '🎦', '📶', '🈁', '🔣', 'ℹ️', '🔤', '🔡', '🔠', '🔢', '🔟', '🔢', '🔠', '🔡', '🔤', 'ℹ️', '🔣', '🈁', '📶', '🎦', '🚮', '🚻', '🚼', '🚺', '🚹', '🛅', '🛄', '🛃', '🛂', '🈂️', '🈳', '🅿️', '♿', '🚾', '🏧', '💤', '🌀', 'Ⓜ️', '💠', '🌐', '❎', '✳️', '❇️', '💹', '🈯', '✅', '♻️', '🔰', '⚜️', '🔱', '🚸', '⚠️', '〽️', '🔆', '🔅', '⁉️', '‼️', '❔', '❕', '❓', '❗', '🚭', '📵', '🔞', '🚱', '🚳', '🚯', '🚷', '♨️', '💢', '💯', '🚫', '📛', '⛔', '🛑', '⭕', '❌', '🆘', '🅾️', '🆑', '🆎', '🅱️', '🅰️', '🈲', '🈹', '🈵', '🈴', '㊗️', '㊙️', '🉐', '💮', '🆚', '✴️', '🈷️', '🈺', '🈸', '🈚', '🈶', '📳', '📴', '☣️', '☢️', '🉑', '⚛️', '🆔', '♓', '♒', '♑', '♐', '♏', '♎', '♍', '♌', '♋', '♊', '♉', '♈', '⛎', '🛐', '☦️', '☮️', '☪️', '✝️', '☮️', '💟', '💝', '💘', '💖', '💗', '💓', '💞', '💕', '❣️', '💔', '🤎', '🤍', '🖤', '💜', '💙', '💚', '💛', '🧡', '❤️',
                                // Flags
                                '🏳️', '🏴', '🏁', '🚩', '🏳️‍🌈', '🏳️‍⚧️', '🇦🇨', '🇦🇩', '🇦🇪', '🇦🇫', '🇦🇬', '🇦🇮', '🇦🇱', '🇦🇲', '🇦🇴', '🇦🇶', '🇦🇷', '🇦🇸', '🇦🇹', '🇦🇺', '🇦🇼', '🇦🇽', '🇦🇿', '🇧🇦', '🇧🇧', '🇧🇩', '🇧🇪', '🇧🇫', '🇧🇬', '🇧🇭', '🇧🇮', '🇧🇯', '🇧🇱', '🇧🇲', '🇧🇳', '🇧🇴', '🇧🇶', '🇧🇷', '🇧🇸', '🇧🇹', '🇧🇻', '🇧🇼', '🇧🇾', '🇧🇿', '🇨🇦', '🇨🇨', '🇨🇩', '🇨🇫', '🇨🇬', '🇨🇭', '🇨🇮', '🇨🇰', '🇨🇱', '🇨🇲', '🇨🇳', '🇨🇴', '🇨🇵', '🇨🇷', '🇨🇺', '🇨🇻', '🇨🇼', '🇨🇽', '🇨🇾', '🇨🇿', '🇩🇪', '🇩🇬', '🇩🇯', '🇩🇰', '🇩🇲', '🇩🇴', '🇩🇿', '🇪🇦', '🇪🇨', '🇪🇪', '🇪🇬', '🇪🇭', '🇪🇷', '🇪🇸', '🇪🇹', '🇪🇺', '🇫🇮', '🇫🇯', '🇫🇰', '🇫🇲', '🇫🇴', '🇫🇷', '🇬🇦', '🇬🇧', '🇬🇩', '🇬🇪', '🇬🇫', '🇬🇬', '🇬🇭', '🇬🇮', '🇬🇱', '🇬🇲', '🇬🇳', '🇬🇵', '🇬🇶', '🇬🇷', '🇬🇸', '🇬🇹', '🇬🇺', '🇬🇼', '🇬🇾', '🇭🇰', '🇭🇲', '🇭🇳', '🇭🇷', '🇭🇹', '🇭🇺', '🇮🇨', '🇮🇩', '🇮🇪', '🇮🇱', '🇮🇲', '🇮🇳', '🇮🇴', '🇮🇶', '🇮🇷', '🇮🇸', '🇮🇹', '🇯🇪', '🇯🇲', '🇯🇴', '🇯🇵', '🇰🇪', '🇰🇬', '🇰🇭', '🇰🇮', '🇰🇲', '🇰🇳', '🇰🇵', '🇰🇷', '🇰🇼', '🇰🇾', '🇰🇿', '🇱🇦', '🇱🇧', '🇱🇨', '🇱🇮', '🇱🇰', '🇱🇷', '🇱🇸', '🇱🇹', '🇱🇺', '🇱🇻', '🇱🇾', '🇲🇦', '🇲🇨', '🇲🇩', '🇲🇪', '🇲🇫', '🇲🇬', '🇲🇭', '🇲🇰', '🇲🇱', '🇲🇲', '🇲🇳', '🇲🇴', '🇲🇵', '🇲🇶', '🇲🇷', '🇲🇸', '🇲🇹', '🇲🇺', '🇲🇻', '🇲🇼', '🇲🇽', '🇲🇾', '🇲🇿', '🇳🇦', '🇳🇨', '🇳🇪', '🇳🇫', '🇳🇬', '🇳🇮', '🇳🇱', '🇳🇴', '🇳🇵', '🇳🇷', '🇳🇺', '🇳🇿', '🇴🇲', '🇵🇦', '🇵🇪', '🇵🇫', '🇵🇬', '🇵🇭', '🇵🇰', '🇵🇱', '🇵🇲', '🇵🇳', '🇵🇷', '🇵🇸', '🇵🇹', '🇵🇼', '🇵🇾', '🇶🇦', '🇷🇪', '🇷🇴', '🇷🇸', '🇷🇺', '🇷🇼', '🇸🇦', '🇸🇧', '🇸🇨', '🇸🇩', '🇸🇪', '🇸🇬', '🇸🇭', '🇸🇮', '🇸🇯', '🇸🇰', '🇸🇱', '🇸🇲', '🇸🇳', '🇸🇴', '🇸🇷', '🇸🇸', '🇸🇹', '🇸🇻', '🇸🇽', '🇸🇾', '🇸🇿', '🇹🇦', '🇹🇨', '🇹🇩', '🇹🇫', '🇹🇬', '🇹🇭', '🇹🇯', '🇹🇰', '🇹🇱', '🇹🇲', '🇹🇳', '🇹🇴', '🇹🇷', '🇹🇹', '🇹🇻', '🇹🇼', '🇹🇿', '🇺🇦', '🇺🇬', '🇺🇲', '🇺🇳', '🇺🇸', '🇺🇾', '🇺🇿', '🇻🇦', '🇻🇨', '🇻🇪', '🇻🇬', '🇻🇮', '🇻🇳', '🇻🇺', '🇼🇫', '🇼🇸', '🇽🇰', '🇾🇪', '🇾🇹', '🇿🇦', '🇿🇲', '🇿🇼', '🏴‍☠️',
                                // Content & Writing (existing)
                                '📝', '✍️', '📰', '📄', '📚', '📖', '📑', '📋', '🗞️', '📰',
                                // Ideas & Information (existing)
                                '💡', 'ℹ️', '🔍', '💭', '🧠', '💬', '🔎', '📌', '📍',
                                // Social & Communication (existing)
                                '📢', '🔗', '🌐', '👥', '💬', '📱', '💻', '🖥️', '⌨️', '🖱️',
                                // Status & Actions (existing)
                                '✅', '⭐', '🔥', '⚡', '🚀', '🎯', '💪', '🎉', '👍', '👏', '🙌',
                                // Media & Visual (existing)
                                '📸', '📹', '🎬', '🎨', '🖼️', '📊', '📈', '📉', '📺', '🎥',
                                // Business & Entrepreneurship (existing)
                                '💼', '🏆', '🎁', '💰', '💵', '💶', '💷', '💴', '💳', '📊', '📈', '📉',
                                '🏢', '🏭', '🏪', '🛒', '🛍️', '📦', '📮', '✉️', '📧', '📨',
                                // IT & SEO & Tech (existing)
                                '💻', '🖥️', '⌨️', '🖱️', '📱', '📲', '🔌', '⚡', '🔋', '💾', '💿', '📀',
                                '🌐', '🔗', '🔐', '🔒', '🔓', '🛡️', '⚙️', '🔧', '🛠️', '📡', '📶',
                                '🔍', '🔎', '📊', '📈', '📉', '🎯', '🎪', '🎨', '🖼️',
                                // Medical & Health (existing)
                                '⚕️', '🏥', '💊', '💉', '🩺', '🩹', '🧬', '🔬', '🧪', '🌡️',
                                '❤️', '💚', '💙', '🧠', '🫀', '🫁', '🦷', '🦴',
                                // Construction & Real Estate (existing)
                                '🏗️', '🏘️', '🏠', '🏡', '🏢', '🏬', '🏭', '🏛️', '🏟️', '🏰',
                                '🔨', '🪚', '🔧', '🛠️', '⚒️', '🪓', '⛏️', '🪚', '🔩', '⚙️',
                                // News & Blog & Publishing (existing)
                                '📰', '🗞️', '📄', '📑', '📋', '📝', '✍️', '📚', '📖', '📗', '📘', '📙',
                                '📕', '📓', '📔', '📒', '📃', '📜', '📰', '🗞️',
                                // Modern & Trending (existing)
                                '✨', '🌟', '💫', '🆕', '🔄', '⚙️', '🎊', '🎈', '🎁', '🎉',
                                '🔥', '💯', '🎯', '⚡', '🚀', '💪', '👑', '🏅', '🥇', '🥈', '🥉',
                            );
                            
                            // Initial batch size (30 emojis).
                            $initial_batch = 30;
                            $total_emojis = count( $emojis );
                            
                            // Display initial batch.
                            for ( $i = 0; $i < min( $initial_batch, $total_emojis ); $i++ ) {
                                echo '<button type="button" class="button fsb-emoji-btn" data-emoji="' . esc_attr( $emojis[ $i ] ) . '" style="margin:2px; font-size:16px; padding:4px 8px;">' . esc_html( $emojis[ $i ] ) . '</button>';
                            }
                            
                            // Store all emojis in data attribute for JavaScript.
                            $emojis_json = wp_json_encode( $emojis );
                            ?>
                            <div id="fsb-emoji-hidden" data-emojis='<?php echo esc_attr( $emojis_json ); ?>' data-shown="<?php echo esc_attr( $initial_batch ); ?>" style="display:none;"></div>
                            <?php if ( $total_emojis > $initial_batch ) : ?>
                                <p style="margin-top:10px;">
                                    <button type="button" id="fsb-load-more-emojis" class="button">
                                        <?php echo esc_html( __( 'Load More Emojis', '5th-social-bot' ) ); ?>
                                        <span id="fsb-emoji-count">(<?php echo esc_html( $total_emojis - $initial_batch ); ?> remaining)</span>
                                    </button>
                                </p>
                            <?php endif; ?>
                        </div>
                    </td>
                </tr>

                <tr>
                    <th scope="row">
                        <label for="fsb_readmore_texts">
                            <?php echo esc_html( __( 'Texts for {readmore_text}', '5th-social-bot' ) ); ?>
                        </label>
                    </th>
                    <td>
                        <textarea id="fsb_readmore_texts" name="fsb_readmore_texts" class="large-text code" rows="5"><?php echo esc_textarea( $settings['readmore_texts'] ); ?></textarea>
                        <p class="description">
                            <?php echo esc_html( __( 'One text per line. For each post, one line is chosen at random. Example: "Read more at:", "More details at:".', '5th-social-bot' ) ); ?>
                        </p>
                    </td>
                </tr>
            </table>

            <p class="submit">
                <button type="submit" class="button button-primary">
                    <?php echo esc_html( __( 'Save settings', '5th-social-bot' ) ); ?>
                </button>
            </p>
        </form>
        </div>

        <div class="fsb-card">
            <h2><?php echo esc_html( __( 'Template preview', '5th-social-bot' ) ); ?></h2>
            <p class="description">
                <?php echo esc_html__( 'Generate a preview of how your Mastodon post will look using the current template and the latest published post/page.', '5th-social-bot' ); ?>
            </p>
            <form method="post" action="" id="fsb-preview-form" style="margin-bottom:1em;">
                <?php wp_nonce_field( 'fifth_social_bot_settings' ); ?>
                <input type="hidden" name="fifth_social_bot_action" value="preview_template" />
                <input type="hidden" name="fsb_template" id="fsb_template_preview_field" value="" />
                <input type="hidden" name="fsb_readmore_texts" id="fsb_readmore_texts_preview_field" value="" />
                <p>
                    <button type="submit" class="button">
                        <?php echo esc_html( __( 'Generate preview (latest post/page)', '5th-social-bot' ) ); ?>
                    </button>
                </p>
            </form>

            <?php if ( ! empty( $template_preview ) ) : ?>
                <h3><?php echo esc_html( __( 'Example Mastodon message (preview)', '5th-social-bot' ) ); ?></h3>
                <textarea class="large-text code" rows="5" readonly><?php echo esc_textarea( $template_preview ); ?></textarea>
                <p class="description">
                    <?php echo esc_html__( 'This is a preview based on the latest published post/page. In production, messages are automatically trimmed to a maximum of 500 characters.', '5th-social-bot' ); ?>
                </p>
            <?php endif; ?>
        </div>

        <div class="fsb-card">
            <h2><?php echo esc_html( __( 'Connection test', '5th-social-bot' ) ); ?></h2>
        <form method="post" action="" style="margin-bottom:1em;">
            <?php wp_nonce_field( 'fifth_social_bot_settings' ); ?>
            <input type="hidden" name="fifth_social_bot_action" value="check_connection" />
            <p>
                <button type="submit" class="button">
                    <?php echo esc_html( __( 'Check connection with Mastodon', '5th-social-bot' ) ); ?>
                </button>
            </p>
        </form>

            <h2><?php echo esc_html( __( 'Test post', '5th-social-bot' ) ); ?></h2>
            <form method="post" action="" id="fsb-test-post-form" style="margin-bottom:1em;">
            <?php wp_nonce_field( 'fifth_social_bot_settings' ); ?>
            <input type="hidden" name="fifth_social_bot_action" value="test_post" />
            <p>
                <button type="submit" class="button button-secondary" id="fsb-test-post-btn">
                    <?php echo esc_html( __( 'Send test post to Mastodon', '5th-social-bot' ) ); ?>
                </button>
            </p>
            <p class="description">
                <?php echo esc_html__( 'Send a test post to verify your Mastodon configuration is working correctly. The test post will be visible on your Mastodon profile.', '5th-social-bot' ); ?>
            </p>
        </form>

            <?php if ( ! empty( $connection_message ) ) : ?>
                <div class="fsb-info-box">
                    <p><?php echo esc_html( $connection_message ); ?></p>
                </div>
            <?php endif; ?>
        </div>

        <p style="margin-top: 2em; font-size: 12px; color: #666;">
            <?php
            printf(
                /* translators: %1$s: link to social.5th.ro, %2$s: link to plugin.5th.ro */
                wp_kses_post( __( 'Plugin created for WordPress → Mastodon (%1$s) integration. Official website: %2$s', '5th-social-bot' ) ),
                '<a href="https://social.5th.ro" target="_blank" rel="noopener noreferrer">social.5th.ro</a>',
                '<a href="https://plugin.5th.ro" target="_blank" rel="noopener noreferrer">plugin.5th.ro</a>'
            );
            ?>
            <br />
            <?php
            echo wp_kses_post(
                'Signed: <strong>SC ARC MEDIASOFT SRL</strong> — <a href="https://e-agentie.ro" target="_blank" rel="noopener noreferrer">e-agentie.ro</a>'
            );
            ?>
        </p>

        <script>
        (function(){
            // Toggle token visibility.
            var tokenInput = document.getElementById('fsb_access_token');
            var toggleBtn = document.getElementById('fsb-toggle-token');
            var toggleText = document.getElementById('fsb-toggle-token-text');
            
            if (tokenInput && toggleBtn && toggleText) {
                toggleBtn.addEventListener('click', function(e){
                    e.preventDefault();
                    if (tokenInput.type === 'password') {
                        tokenInput.type = 'text';
                        toggleText.textContent = '<?php echo esc_js( __( 'Hide', '5th-social-bot' ) ); ?>';
                    } else {
                        tokenInput.type = 'password';
                        toggleText.textContent = '<?php echo esc_js( __( 'Show', '5th-social-bot' ) ); ?>';
                    }
                });
            }

            // Toggle instance URL field visibility.
            var useCustomInstance = document.getElementById('fsb_use_custom_instance');
            var instanceUrlRow = document.getElementById('fsb_instance_url_row');
            
            if (useCustomInstance && instanceUrlRow) {
                function toggleInstanceField() {
                    if (useCustomInstance.checked) {
                        instanceUrlRow.style.display = '';
                    } else {
                        instanceUrlRow.style.display = 'none';
                    }
                }
                
                useCustomInstance.addEventListener('change', toggleInstanceField);
                // Run on page load
                toggleInstanceField();
            }
            
            // Emoji insert.
            var tpl = document.getElementById('fsb_template');
            var emojiButtons = document.querySelectorAll('.fsb-emoji-btn');
            if (tpl && emojiButtons.length) {
                emojiButtons.forEach(function(btn){
                    btn.addEventListener('click', function(e){
                        e.preventDefault();
                        var emoji = this.getAttribute('data-emoji') || '';
                        if (!emoji) return;

                        if (typeof tpl.selectionStart === 'number') {
                            var start = tpl.selectionStart;
                            var end   = tpl.selectionEnd;
                            var val   = tpl.value;
                            tpl.value = val.substring(0, start) + emoji + val.substring(end);
                            tpl.selectionStart = tpl.selectionEnd = start + emoji.length;
                        } else {
                            tpl.value += emoji;
                        }
                        tpl.focus();
                    });
                });
            }
            
            // Load More Emojis - progressive loading (doubles batch each time).
            var loadMoreBtn = document.getElementById('fsb-load-more-emojis');
            var emojiContainer = document.getElementById('fsb-emoji-container');
            var emojiHidden = document.getElementById('fsb-emoji-hidden');
            var emojiCountSpan = document.getElementById('fsb-emoji-count');
            
            if (loadMoreBtn && emojiContainer && emojiHidden) {
                loadMoreBtn.addEventListener('click', function(e){
                    e.preventDefault();
                    var allEmojis = JSON.parse(emojiHidden.getAttribute('data-emojis') || '[]');
                    var shown = parseInt(emojiHidden.getAttribute('data-shown') || '30', 10);
                    var total = allEmojis.length;
                    
                    // Double the batch size each time.
                    var newBatchSize = shown * 2;
                    var toShow = Math.min(newBatchSize, total);
                    
                    // Add new emojis.
                    for (var i = shown; i < toShow; i++) {
                        var btn = document.createElement('button');
                        btn.type = 'button';
                        btn.className = 'button fsb-emoji-btn';
                        btn.setAttribute('data-emoji', allEmojis[i]);
                        btn.style.cssText = 'margin:2px; font-size:16px; padding:4px 8px;';
                        btn.textContent = allEmojis[i];
                        
                        // Attach click handler (same as existing emoji buttons).
                        btn.addEventListener('click', function(e){
                            e.preventDefault();
                            var emoji = this.getAttribute('data-emoji') || '';
                            if (!emoji) return;
                            
                            if (typeof tpl.selectionStart === 'number') {
                                var start = tpl.selectionStart;
                                var end   = tpl.selectionEnd;
                                var val   = tpl.value;
                                tpl.value = val.substring(0, start) + emoji + val.substring(end);
                                tpl.selectionStart = tpl.selectionEnd = start + emoji.length;
                            } else {
                                tpl.value += emoji;
                            }
                            tpl.focus();
                        });
                        
                        // Insert before the "Load More" button container.
                        var loadMoreContainer = loadMoreBtn.parentElement;
                        emojiContainer.insertBefore(btn, loadMoreContainer);
                    }
                    
                    // Update shown count.
                    emojiHidden.setAttribute('data-shown', toShow);
                    
                    // Update or hide button.
                    if (toShow >= total) {
                        loadMoreBtn.parentElement.style.display = 'none';
                    } else {
                        if (emojiCountSpan) {
                            emojiCountSpan.textContent = '(' + (total - toShow) + ' remaining)';
                        }
                    }
                });
            }

            // Preview form: copy current template/readmore into hidden fields.
            var previewForm = document.getElementById('fsb-preview-form');
            if (previewForm && tpl) {
                previewForm.addEventListener('submit', function(){
                    var tplPreview = document.getElementById('fsb_template_preview_field');
                    var rmPreview  = document.getElementById('fsb_readmore_texts_preview_field');
                    if (tplPreview) {
                        tplPreview.value = tpl.value;
                    }
                    var rm = document.getElementById('fsb_readmore_texts');
                    if (rm && rmPreview) {
                        rmPreview.value = rm.value;
                    }
                });
            }

            // Reset template to default.
            var resetBtn = document.getElementById('fsb-reset-template');
            if (resetBtn && tpl) {
                resetBtn.addEventListener('click', function(e){
                    e.preventDefault();
                    if (confirm('<?php echo esc_js( __( 'Are you sure you want to reset the template to default? This will replace the current template.', '5th-social-bot' ) ); ?>')) {
                        var form = document.getElementById('fsb-settings-form');
                        var actionField = document.getElementById('fsb-action-field');
                        if (form && actionField) {
                            actionField.value = 'reset_template';
                            form.submit();
                        }
                    }
                });
            }
            
            // Loading states for buttons.
            var forms = document.querySelectorAll('form[method="post"]');
            forms.forEach(function(form) {
                form.addEventListener('submit', function(e) {
                    var submitBtn = form.querySelector('button[type="submit"]');
                    if (submitBtn && !submitBtn.disabled) {
                        var originalText = submitBtn.textContent;
                        submitBtn.disabled = true;
                        submitBtn.style.opacity = '0.6';
                        submitBtn.style.cursor = 'wait';
                        
                        // Add spinner if not already present.
                        if (!submitBtn.querySelector('.spinner')) {
                            var spinner = document.createElement('span');
                            spinner.className = 'spinner';
                            spinner.style.cssText = 'display:inline-block; margin-right:5px; vertical-align:middle;';
                            spinner.innerHTML = '⏳';
                            submitBtn.insertBefore(spinner, submitBtn.firstChild);
                        }
                        
                        // Re-enable after 5 seconds as fallback (in case of error).
                        setTimeout(function() {
                            if (submitBtn.disabled) {
                                submitBtn.disabled = false;
                                submitBtn.style.opacity = '1';
                                submitBtn.style.cursor = 'pointer';
                                var spinner = submitBtn.querySelector('.spinner');
                                if (spinner) {
                                    spinner.remove();
                                }
                            }
                        }, 5000);
                    }
                });
            });
        })();
        </script>
        <?php
    }

    /**
     * Bulk & queue tab.
     */
    protected function render_bulk_tab() {
        $base_url = add_query_arg(
            array(
                'page' => 'fifth-social-bot',
                'tab'  => 'bulk',
            ),
            admin_url( 'options-general.php' )
        );

        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- read-only pagination param.
        $paged    = isset( $_GET['fsb_paged'] ) ? max( 1, (int) $_GET['fsb_paged'] ) : 1;
        $per_page = 20;

        $args = array(
            'post_type'      => array( 'post', 'page' ),
            'post_status'    => 'publish',
            'posts_per_page' => $per_page,
            'paged'          => $paged,
            'orderby'        => 'date',
            'order'          => 'DESC',
        );

        $query = new WP_Query( $args );
        $queue = Fifth_Social_Bot_Queue::get_queue();
        ?>
        <div class="fsb-card">
            <h2><?php echo esc_html( __( 'Bulk posting', '5th-social-bot' ) ); ?></h2>
        <p>
            <?php echo esc_html( __( 'You can quickly add the latest post or batches of 20 / 40 / 60 / 80 to the queue. Posts in the queue are sent gradually via WP-Cron (1 post every 5 minutes) to avoid spam.', '5th-social-bot' ) ); ?>
        </p>
        <p class="description">
            <strong><?php echo esc_html__( 'How it works:', '5th-social-bot' ); ?></strong>
            <br />
            <?php echo esc_html__( 'When you click "Post last 20" (or any batch), the plugin adds those posts to an internal queue. WP-Cron then processes the queue automatically: it sends 1 post every 5 minutes until the queue is empty. This prevents rate limiting and spam issues on Mastodon.', '5th-social-bot' ); ?>
            <br />
            <?php echo esc_html__( 'Posts that are already marked as "Posted to Mastodon" or are already in the queue are automatically skipped.', '5th-social-bot' ); ?>
            <br />
            <?php echo esc_html__( 'You can see the current queue status below. The queue persists even if you close this page.', '5th-social-bot' ); ?>
        </p>

            <form method="post" action="" style="margin-bottom:1em;">
                <?php wp_nonce_field( 'fifth_social_bot_settings' ); ?>
                <input type="hidden" name="fifth_social_bot_action" value="bulk_last" />
                <input type="hidden" name="fsb_bulk_count" id="fsb_bulk_count" value="0" />

                <div class="fsb-button-group">
                    <button type="submit" class="button" onclick="document.getElementById('fsb_bulk_count').value='1';">
                        <?php echo esc_html( __( 'Post last article', '5th-social-bot' ) ); ?>
                    </button>
                    <button type="submit" class="button" onclick="document.getElementById('fsb_bulk_count').value='20';">
                        <?php echo esc_html( __( 'Post last 20', '5th-social-bot' ) ); ?>
                    </button>
                    <button type="submit" class="button" onclick="document.getElementById('fsb_bulk_count').value='40';">
                        <?php echo esc_html( __( 'Post last 40', '5th-social-bot' ) ); ?>
                    </button>
                    <button type="submit" class="button" onclick="document.getElementById('fsb_bulk_count').value='60';">
                        <?php echo esc_html( __( 'Post last 60', '5th-social-bot' ) ); ?>
                    </button>
                    <button type="submit" class="button" onclick="document.getElementById('fsb_bulk_count').value='80';">
                        <?php echo esc_html( __( 'Post last 80', '5th-social-bot' ) ); ?>
                    </button>
                </div>
            </form>
        </div>

        <div class="fsb-card">
            <h2><?php echo esc_html( __( 'Latest posts/pages', '5th-social-bot' ) ); ?></h2>
        <table class="widefat fixed striped">
            <thead>
                <tr>
                    <th><?php echo esc_html( __( 'Title', '5th-social-bot' ) ); ?></th>
                    <th><?php echo esc_html( __( 'Type', '5th-social-bot' ) ); ?></th>
                    <th><?php echo esc_html( __( 'Date', '5th-social-bot' ) ); ?></th>
                    <th><?php echo esc_html( __( 'Mastodon status', '5th-social-bot' ) ); ?></th>
                    <th><?php echo esc_html( __( 'Actions', '5th-social-bot' ) ); ?></th>
                </tr>
            </thead>
            <tbody>
            <?php
            if ( $query->have_posts() ) :
                foreach ( $query->posts as $post ) :
                    $post_id   = $post->ID;
                    $posted    = ( '1' === get_post_meta( $post_id, '_fifth_social_bot_posted', true ) );
                    $queued    = in_array( $post_id, $queue, true ) || ( '1' === get_post_meta( $post_id, '_fifth_social_bot_queued', true ) );
                    $post_type = get_post_type_object( $post->post_type );
                    ?>
                    <tr>
                        <td>
                            <a href="<?php echo esc_url( get_edit_post_link( $post_id ) ); ?>">
                                <?php echo esc_html( get_the_title( $post_id ) ); ?>
                            </a>
                        </td>
                        <td><?php echo esc_html( $post_type ? $post_type->labels->singular_name : $post->post_type ); ?></td>
                        <td><?php echo esc_html( get_the_date( 'Y-m-d H:i', $post_id ) ); ?></td>
                        <td>
                            <?php
                            if ( $posted ) {
                                $mastodon_url = get_post_meta( $post_id, '_fifth_social_bot_mastodon_url', true );
                                $boost_url = get_post_meta( $post_id, '_fifth_social_bot_boost_url', true );
                                $boost_queued = get_post_meta( $post_id, '_fifth_social_bot_boost_queued', true );
                                $boost_next_time = get_post_meta( $post_id, '_fifth_social_bot_boost_next_time', true );
                                
                                if ( ! empty( $mastodon_url ) ) {
                                    echo '<span style="color:green;">' . esc_html( __( 'Posted to Mastodon', '5th-social-bot' ) ) . '</span>';
                                    echo '<br />';
                                    echo '<a href="' . esc_url( $mastodon_url ) . '" target="_blank" rel="noopener noreferrer" style="font-size:12px; color:#0073aa;">' . esc_html( __( 'View post', '5th-social-bot' ) ) . '</a>';
                                    
                                    // Boost status.
                                    if ( ! empty( $boost_url ) ) {
                                        echo '<br />';
                                        echo '<span style="color:green;">✓ ' . esc_html( __( 'Boosted', '5th-social-bot' ) ) . '</span>';
                                        echo '<br />';
                                        echo '<a href="' . esc_url( $boost_url ) . '" target="_blank" rel="noopener noreferrer" style="font-size:12px; color:#d98500;">' . esc_html( __( 'View boost', '5th-social-bot' ) ) . '</a>';
                                    } elseif ( ! empty( $boost_queued ) ) {
                                        echo '<br />';
                                        echo '<span style="color:#d98500;">⏳ ' . esc_html( __( 'Boost queued', '5th-social-bot' ) ) . '</span>';
                                        if ( ! empty( $boost_next_time ) ) {
                                            $next_time = (int) $boost_next_time;
                                            $current_time = time();
                                            if ( $next_time > $current_time ) {
                                                $remaining = $next_time - $current_time;
                                                $remaining_minutes = ceil( $remaining / 60 );
                                                echo '<br />';
                                                echo '<span style="font-size:11px; color:#666;">' . sprintf(
                                                    /* translators: %d: number of minutes */
                                                    esc_html__( 'Next boost in ~%d min', '5th-social-bot' ),
                                                    absint( $remaining_minutes )
                                                ) . '</span>';
                                            }
                                        }
                                    } else {
                                        // Not boosted - status will be shown in Actions column.
                                        // Don't show anything here to avoid duplication.
                                    }
                                } else {
                                    echo '<span style="color:green;">' . esc_html( __( 'Posted to Mastodon', '5th-social-bot' ) ) . '</span>';
                                }
                            } elseif ( $queued ) {
                                echo '<span style="color:#d98500;">' . esc_html( __( 'Queued', '5th-social-bot' ) ) . '</span>';
                            } else {
                                echo '<span style="color:#999;">' . esc_html( __( 'Not posted', '5th-social-bot' ) ) . '</span>';
                            }
                            ?>
                        </td>
                        <td>
                            <?php if ( ! $posted ) : ?>
                                <form method="post" action="" style="display:inline;">
                                    <?php wp_nonce_field( 'fifth_social_bot_settings' ); ?>
                                    <input type="hidden" name="fifth_social_bot_action" value="post_single" />
                                    <input type="hidden" name="fsb_post_id" value="<?php echo esc_attr( $post_id ); ?>" />
                                    <button type="submit" class="button button-small">
                                        <?php echo esc_html( __( 'Post now', '5th-social-bot' ) ); ?>
                                    </button>
                                </form>
                                <br />
                                <?php
                                // Queue reorder buttons.
                                $queue = Fifth_Social_Bot_Queue::get_queue();
                                $queue_index = array_search( $post_id, $queue, true );
                                if ( false !== $queue_index ) :
                                    ?>
                                    <div style="margin-top:5px;">
                                        <?php if ( $queue_index > 0 ) : ?>
                                            <form method="post" action="" style="display:inline;">
                                                <?php wp_nonce_field( 'fifth_social_bot_settings' ); ?>
                                                <input type="hidden" name="fifth_social_bot_action" value="move_queue_up" />
                                                <input type="hidden" name="fsb_post_id" value="<?php echo esc_attr( $post_id ); ?>" />
                                                <button type="submit" class="button button-small" title="<?php echo esc_attr__( 'Move up in queue', '5th-social-bot' ); ?>">↑</button>
                                            </form>
                                        <?php endif; ?>
                                        <?php if ( $queue_index < count( $queue ) - 1 ) : ?>
                                            <form method="post" action="" style="display:inline;">
                                                <?php wp_nonce_field( 'fifth_social_bot_settings' ); ?>
                                                <input type="hidden" name="fifth_social_bot_action" value="move_queue_down" />
                                                <input type="hidden" name="fsb_post_id" value="<?php echo esc_attr( $post_id ); ?>" />
                                                <button type="submit" class="button button-small" title="<?php echo esc_attr__( 'Move down in queue', '5th-social-bot' ); ?>">↓</button>
                                            </form>
                                        <?php endif; ?>
                                    </div>
                                <?php endif; ?>
                            <?php else : ?>
                                <?php
                                // Actions for posted items: Boost now button if posted but not boosted.
                                $mastodon_url = get_post_meta( $post_id, '_fifth_social_bot_mastodon_url', true );
                                $boost_url = get_post_meta( $post_id, '_fifth_social_bot_boost_url', true );
                                $boost_queued = get_post_meta( $post_id, '_fifth_social_bot_boost_queued', true );
                                $settings = Fifth_Social_Bot_Core::get_settings();
                                
                                // Show "Boost now" button if: posted to Mastodon, boost enabled, not already boosted, boost account configured.
                                // Allow button even if in queue (to force boost).
                                if ( ! empty( $mastodon_url ) && empty( $boost_url ) && ! empty( $settings['boost_enabled'] ) && ! empty( $settings['boost_instance'] ) && ! empty( $settings['boost_access_token'] ) ) :
                                    ?>
                                    <form method="post" action="" style="display:inline;">
                                        <?php wp_nonce_field( 'fifth_social_bot_settings' ); ?>
                                        <input type="hidden" name="fifth_social_bot_action" value="boost_single" />
                                        <input type="hidden" name="fsb_post_id" value="<?php echo esc_attr( $post_id ); ?>" />
                                        <button type="submit" class="button button-small">
                                            <?php echo esc_html( __( 'Boost now', '5th-social-bot' ) ); ?>
                                        </button>
                                    </form>
                                    <?php
                                endif;
                                
                                // Don't show boost queue reorder buttons here - they are only in "Boost Queue & Statistics" section.
                                if ( empty( $mastodon_url ) || empty( $settings['boost_enabled'] ) || empty( $settings['boost_instance'] ) || empty( $settings['boost_access_token'] ) ) :
                                    ?>
                                    <em>—</em>
                                <?php endif; ?>
                            <?php endif; ?>
                        </td>
                    </tr>
                    <?php
                endforeach;
            else :
                ?>
                <tr>
                    <td colspan="5"><?php echo esc_html( __( 'No posts found.', '5th-social-bot' ) ); ?></td>
                </tr>
            <?php endif; ?>
            </tbody>
        </table>

        <?php if ( $query->max_num_pages > 1 ) : ?>
            <div class="tablenav">
                <div class="tablenav-pages">
                    <?php
                    $page_links = paginate_links(
                        array(
                            'base'      => add_query_arg( 'fsb_paged', '%#%', $base_url ),
                            'format'    => '',
                            'current'   => max( 1, $paged ),
                            'total'     => $query->max_num_pages,
                            'prev_text' => '&laquo;',
                            'next_text' => '&raquo;',
                        )
                    );
                    echo wp_kses_post( $page_links );
                    ?>
                </div>
            </div>
        <?php endif; ?>

        <?php
        wp_reset_postdata();
        $queue = Fifth_Social_Bot_Queue::get_queue();
        ?>
        <hr />
        <h2><?php echo esc_html( __( 'Current queue', '5th-social-bot' ) ); ?></h2>
            <?php
            // Check boost queue as well.
            $boost_queue = Fifth_Social_Bot_Boost::get_boost_queue();
            $has_boost_queue = ! empty( $boost_queue );
            ?>
            <?php if ( empty( $queue ) && ! $has_boost_queue ) : ?>
                <div class="fsb-info-box">
                    <p style="margin: 0;">
                        <strong><?php echo esc_html__( 'Queue is empty', '5th-social-bot' ); ?></strong>
                        <br />
                        <?php echo esc_html__( 'No posts are currently waiting to be posted to Mastodon. Use the buttons above to add posts to the queue.', '5th-social-bot' ); ?>
                    </p>
                </div>
        <?php else : ?>
            <?php if ( ! empty( $queue ) ) : ?>
            <p>
                <?php
                printf(
                    /* translators: %d: number of posts in queue */
                    esc_html__( '%d posts in queue.', '5th-social-bot' ),
                    (int) count( $queue )
                );
                ?>
            </p>
                <p style="max-width:600px;">IDs: <?php echo esc_html( implode( ', ', $queue ) ); ?></p>
                <p class="description">
                    <?php
                    $queue_size = count( $queue );
                    $estimated_minutes = $queue_size * 5; // 5 minutes per post.
                    $estimated_hours = floor( $estimated_minutes / 60 );
                    $estimated_remaining_minutes = $estimated_minutes % 60;
                    
                    if ( $estimated_hours > 0 ) {
                        printf(
                            /* translators: %1$d: hours, %2$d: minutes */
                            esc_html__( 'Estimated time to process all queued posts: approximately %1$d hour(s) and %2$d minute(s) (1 post every 5 minutes).', '5th-social-bot' ),
                            absint( $estimated_hours ),
                            absint( $estimated_remaining_minutes )
                        );
                    } else {
                        printf(
                            /* translators: %d: minutes */
                            esc_html__( 'Estimated time to process all queued posts: approximately %d minute(s) (1 post every 5 minutes).', '5th-social-bot' ),
                            absint( $estimated_minutes )
                        );
                    }
                    ?>
                </p>
            <?php endif; ?>
            
            <?php if ( $has_boost_queue ) : ?>
                <p style="margin-top: 20px;">
                    <?php
                    printf(
                        /* translators: %d: number of posts in boost queue */
                        esc_html__( '%d posts in boost queue.', '5th-social-bot' ),
                        (int) count( $boost_queue )
                    );
                    ?>
                </p>
                <p style="max-width:600px;">Boost Queue IDs: <?php echo esc_html( implode( ', ', $boost_queue ) ); ?></p>
                <p class="description">
                    <?php echo esc_html__( 'These posts are waiting to be boosted on the second Mastodon instance. They will be processed automatically according to rate limits.', '5th-social-bot' ); ?>
                </p>
            <?php endif; ?>
        <?php endif; ?>
        
            <?php
            $settings = Fifth_Social_Bot_Core::get_settings();
            if ( ! empty( $settings['boost_enabled'] ) ) :
                $boost_queue = Fifth_Social_Bot_Boost::get_boost_queue();
                $boost_stats = get_option( 'fifth_social_bot_boost_stats', array() );
                $today = gmdate( 'Y-m-d' );
                $last_boost_date = isset( $boost_stats['last_boost_date'] ) ? $boost_stats['last_boost_date'] : '';
                $boost_count_today = isset( $boost_stats['boost_count_today'] ) ? (int) $boost_stats['boost_count_today'] : 0;
                if ( $today !== $last_boost_date ) {
                    $boost_count_today = 0;
                }
                
                $boost_max_per_day = isset( $settings['boost_max_per_day'] ) ? (int) $settings['boost_max_per_day'] : 10;
                $boost_min_interval = isset( $settings['boost_min_interval'] ) ? (int) $settings['boost_min_interval'] : 30;
                
                // Calculate remaining boosts for today.
                $boost_remaining = $boost_max_per_day - $boost_count_today;
                ?>
        </div>

        <div class="fsb-card">
            <h2><?php echo esc_html( __( 'Boost Queue & Statistics', '5th-social-bot' ) ); ?></h2>
            
            <?php
            // Calculate next boost time.
            $next_boost_time = null;
            $next_boost_timestamp = null;
            
            // Check WP-Cron scheduled event.
            $cron_time = wp_next_scheduled( 'fifth_social_bot_process_boost_queue_event' );
            if ( $cron_time && $cron_time > time() ) {
                $next_boost_timestamp = $cron_time;
            }
            
            // Also check the first post in queue for boost_next_time.
            if ( ! empty( $boost_queue ) ) {
                $first_post_id = $boost_queue[0];
                $first_post_next_time = get_post_meta( $first_post_id, '_fifth_social_bot_boost_next_time', true );
                if ( ! empty( $first_post_next_time ) ) {
                    $first_post_next_timestamp = (int) $first_post_next_time;
                    if ( $first_post_next_timestamp > time() ) {
                        // Use the earliest time.
                        if ( null === $next_boost_timestamp || $first_post_next_timestamp < $next_boost_timestamp ) {
                            $next_boost_timestamp = $first_post_next_timestamp;
                        }
                    }
                }
            }
            
            // If no scheduled time found but we have posts in queue and boosts remaining, estimate based on last boost.
            if ( null === $next_boost_timestamp && ! empty( $boost_queue ) && $boost_remaining > 0 ) {
                $last_boost_time = isset( $boost_stats['last_boost_time'] ) ? (int) $boost_stats['last_boost_time'] : 0;
                if ( $last_boost_time > 0 ) {
                    $next_boost_timestamp = $last_boost_time + ( $boost_min_interval * 60 );
                    // If the calculated time is in the past, boost can happen now.
                    if ( $next_boost_timestamp <= time() ) {
                        $next_boost_timestamp = time() + 60; // Next minute.
                    }
                } else {
                    // No previous boost, can boost now.
                    $next_boost_timestamp = time() + 60; // Next minute.
                }
            }
            
            // Calculate time remaining.
            $time_remaining = null;
            if ( null !== $next_boost_timestamp ) {
                $time_remaining = $next_boost_timestamp - time();
                if ( $time_remaining < 0 ) {
                    $time_remaining = 0;
                }
            }
            ?>
            
            <div class="fsb-stats-grid">
                <div class="fsb-stat-card">
                    <div class="stat-value"><?php echo absint( count( $boost_queue ) ); ?></div>
                    <div class="stat-label"><?php echo esc_html__( 'Posts in boost queue', '5th-social-bot' ); ?></div>
                </div>
                <div class="fsb-stat-card">
                    <div class="stat-value"><?php echo absint( $boost_count_today ); ?> / <?php echo absint( $boost_max_per_day ); ?></div>
                    <div class="stat-label"><?php echo esc_html__( 'Boosts today', '5th-social-bot' ); ?></div>
                </div>
                <div class="fsb-stat-card">
                    <div class="stat-value"><?php echo absint( $boost_min_interval ); ?> min</div>
                    <div class="stat-label"><?php echo esc_html__( 'Min interval', '5th-social-bot' ); ?></div>
                </div>
                <?php if ( null !== $time_remaining && ! empty( $boost_queue ) && $boost_remaining > 0 ) : ?>
                <div class="fsb-stat-card">
                    <div class="stat-value" style="color: #d98500;">
                        <?php
                        if ( $time_remaining <= 60 ) {
                            echo esc_html( __( 'Now', '5th-social-bot' ) );
                        } elseif ( $time_remaining < 3600 ) {
                            // Less than 1 hour - show minutes.
                            $minutes = ceil( $time_remaining / 60 );
                            printf(
                                /* translators: %d: minutes */
                                esc_html__( '%d min', '5th-social-bot' ),
                                absint( $minutes )
                            );
                        } else {
                            // 1 hour or more - show hours and minutes.
                            $hours = floor( $time_remaining / 3600 );
                            $minutes = ceil( ( $time_remaining % 3600 ) / 60 );
                            if ( $minutes > 0 ) {
                                printf(
                                    /* translators: %1$d: hours, %2$d: minutes */
                                    esc_html__( '%1$dh %2$dm', '5th-social-bot' ),
                                    absint( $hours ),
                                    absint( $minutes )
                                );
                            } else {
                                printf(
                                    /* translators: %d: hours */
                                    esc_html__( '%d h', '5th-social-bot' ),
                                    absint( $hours )
                                );
                            }
                        }
                        ?>
                    </div>
                    <div class="stat-label"><?php echo esc_html__( 'Next boost in', '5th-social-bot' ); ?></div>
                </div>
                <?php elseif ( ! empty( $boost_queue ) && $boost_remaining <= 0 ) : ?>
                <div class="fsb-stat-card">
                    <div class="stat-value" style="color: #d63638;">
                        <?php
                        // Calculate time until midnight (when limit resets).
                        $midnight = strtotime( 'tomorrow midnight' );
                        $time_until_midnight = $midnight - time();
                        $hours = floor( $time_until_midnight / 3600 );
                        $minutes = ceil( ( $time_until_midnight % 3600 ) / 60 );
                        if ( $hours > 0 ) {
                            printf(
                                /* translators: %1$d: hours, %2$d: minutes */
                                esc_html__( '%1$dh %2$dm', '5th-social-bot' ),
                                absint( $hours ),
                                absint( $minutes )
                            );
                        } else {
                            printf(
                                /* translators: %d: minutes */
                                esc_html__( '%d min', '5th-social-bot' ),
                                absint( $minutes )
                            );
                        }
                        ?>
                    </div>
                    <div class="stat-label"><?php echo esc_html__( 'Limit resets in', '5th-social-bot' ); ?></div>
                </div>
                <?php endif; ?>
            </div>
            <p>
                <?php
                printf(
                    /* translators: %d: number of posts in boost queue */
                    esc_html__( '%d posts waiting for boost.', '5th-social-bot' ),
                    (int) count( $boost_queue )
                );
                ?>
            </p>
            <?php
            // Rate limit warnings.
            if ( $boost_remaining <= 2 && $boost_remaining > 0 ) {
                ?>
                <div class="fsb-info-box warning">
                    <strong>⚠ <?php echo esc_html__( 'Rate limit warning:', '5th-social-bot' ); ?></strong>
                    <?php
                    printf(
                        /* translators: %d: remaining boosts */
                        esc_html__( 'Only %d boost(s) remaining today. The daily limit will reset at midnight.', '5th-social-bot' ),
                        absint( $boost_remaining )
                    );
                    ?>
                </div>
                <?php
            } elseif ( $boost_remaining <= 0 ) {
                ?>
                <div class="fsb-info-box error">
                    <strong>✗ <?php echo esc_html__( 'Daily boost limit reached:', '5th-social-bot' ); ?></strong>
                    <?php echo esc_html__( 'You have reached the maximum number of boosts for today. The limit will reset at midnight. Posts will remain in the boost queue and will be processed automatically when the limit resets.', '5th-social-bot' ); ?>
                </div>
                <?php
            }
            ?>
            <p class="description">
                <?php
                printf(
                    /* translators: %1$d: maximum boosts per day, %2$d: minimum interval in minutes */
                    esc_html__( 'Boost rate limits: Maximum %1$d boosts per day, minimum %2$d minutes between boosts. Posts are automatically boosted when rate limits allow.', '5th-social-bot' ),
                    absint( $boost_max_per_day ),
                    absint( $boost_min_interval )
                );
                ?>
            </p>
            <?php if ( empty( $boost_queue ) ) : ?>
                <div class="fsb-info-box">
                    <p style="margin: 0;">
                        <strong><?php echo esc_html__( 'Boost queue is empty', '5th-social-bot' ); ?></strong>
                        <br />
                        <?php echo esc_html__( 'No posts are currently waiting to be boosted. Posts will be automatically added to the boost queue after they are successfully posted to Mastodon (if Auto-Boost is enabled).', '5th-social-bot' ); ?>
                    </p>
                </div>
            <?php else : ?>
                <table class="widefat fixed striped">
                    <thead>
                        <tr>
                            <th><?php echo esc_html__( 'Post ID', '5th-social-bot' ); ?></th>
                            <th><?php echo esc_html__( 'Title', '5th-social-bot' ); ?></th>
                            <th><?php echo esc_html__( 'Status', '5th-social-bot' ); ?></th>
                            <th><?php echo esc_html__( 'Actions', '5th-social-bot' ); ?></th>
                        </tr>
                    </thead>
                    <tbody>
                    <?php
                    $queue_index = 0;
                    $current_time = time();
                    $last_boost_time = isset( $boost_stats['last_boost_time'] ) ? (int) $boost_stats['last_boost_time'] : 0;
                    $cron_time = wp_next_scheduled( 'fifth_social_bot_process_boost_queue_event' );
                    $next_available_boost_time = null;
                    
                    // Calculate when the next boost can happen.
                    if ( $boost_remaining > 0 ) {
                        if ( $cron_time && $cron_time > $current_time ) {
                            $next_available_boost_time = $cron_time;
                        } elseif ( $last_boost_time > 0 ) {
                            $next_available_boost_time = $last_boost_time + ( $boost_min_interval * 60 );
                            if ( $next_available_boost_time <= $current_time ) {
                                $next_available_boost_time = $current_time + 60; // Can boost now, next minute.
                            }
                        } else {
                            $next_available_boost_time = $current_time + 60; // Can boost now.
                        }
                    } else {
                        // Daily limit reached, calculate when it resets.
                        $midnight = strtotime( 'tomorrow midnight' );
                        $next_available_boost_time = $midnight;
                    }
                    
                    foreach ( $boost_queue as $boost_post_id ) :
                        $boost_post = get_post( $boost_post_id );
                        if ( ! $boost_post ) {
                            continue;
                        }
                        $boost_post_title = get_the_title( $boost_post_id );
                        $mastodon_url = get_post_meta( $boost_post_id, '_fifth_social_bot_mastodon_url', true );
                        $boost_next_time = get_post_meta( $boost_post_id, '_fifth_social_bot_boost_next_time', true );
                        
                        // Calculate when this specific post will be boosted.
                        $post_boost_time = null;
                        if ( ! empty( $boost_next_time ) ) {
                            $post_boost_time = (int) $boost_next_time;
                            if ( $post_boost_time <= $current_time ) {
                                $post_boost_time = null; // Time passed, recalculate.
                            }
                        }
                        
                        // If no specific time set, calculate based on position in queue.
                        if ( null === $post_boost_time && null !== $next_available_boost_time ) {
                            // First post can be boosted at next_available_boost_time.
                            // Subsequent posts need to wait for previous ones + interval.
                            if ( $queue_index === 0 ) {
                                $post_boost_time = $next_available_boost_time;
                            } else {
                                // Each post after the first needs to wait for the previous one + interval.
                                $post_boost_time = $next_available_boost_time + ( $queue_index * $boost_min_interval * 60 );
                            }
                        }
                        
                        // If daily limit is reached, show when limit resets.
                        if ( $boost_remaining <= 0 && null === $post_boost_time ) {
                            $midnight = strtotime( 'tomorrow midnight' );
                            $post_boost_time = $midnight;
                        }
                        ?>
                        <tr>
                            <td><?php echo absint( $boost_post_id ); ?></td>
                            <td><?php echo esc_html( $boost_post_title ); ?></td>
                            <td>
                                <?php if ( ! empty( $mastodon_url ) ) : ?>
                                    <span style="color:green;"><?php echo esc_html__( 'Posted to Mastodon', '5th-social-bot' ); ?></span>
                                    <br />
                                    <a href="<?php echo esc_url( $mastodon_url ); ?>" target="_blank" rel="noopener noreferrer" style="font-size:12px; color:#0073aa;"><?php echo esc_html__( 'View post', '5th-social-bot' ); ?></a>
                                    <?php if ( null !== $post_boost_time ) : ?>
                                        <?php
                                        $remaining = $post_boost_time - $current_time;
                                        if ( $remaining > 0 ) {
                                            ?>
                                            <br />
                                            <span style="color:#d98500; font-size:12px; font-weight:500;">
                                                ⏳ 
                                                <?php
                                                if ( $boost_remaining <= 0 ) {
                                                    // Daily limit reached.
                                                    echo esc_html__( 'Next boost after limit reset in: ', '5th-social-bot' );
                                                } else {
                                                    echo esc_html__( 'Next boost in: ', '5th-social-bot' );
                                                }
                                                
                                                if ( $remaining <= 60 ) {
                                                    echo esc_html__( 'Now', '5th-social-bot' );
                                                } elseif ( $remaining < 3600 ) {
                                                    // Less than 1 hour - show minutes.
                                                    $minutes = ceil( $remaining / 60 );
                                                    printf(
                                                        /* translators: %d: minutes */
                                                        esc_html__( '%d min', '5th-social-bot' ),
                                                        absint( $minutes )
                                                    );
                                                } else {
                                                    // 1 hour or more - show hours and minutes.
                                                    $hours = floor( $remaining / 3600 );
                                                    $minutes = ceil( ( $remaining % 3600 ) / 60 );
                                                    if ( $minutes > 0 ) {
                                                        printf(
                                                            /* translators: %1$d: hours, %2$d: minutes */
                                                            esc_html__( '%1$dh %2$dm', '5th-social-bot' ),
                                                            absint( $hours ),
                                                            absint( $minutes )
                                                        );
                                                    } else {
                                                        printf(
                                                            /* translators: %d: hours */
                                                            esc_html__( '%d h', '5th-social-bot' ),
                                                            absint( $hours )
                                                        );
                                                    }
                                                }
                                                ?>
                                            </span>
                                        <?php } else { ?>
                                            <br />
                                            <span style="color:#00a32a; font-size:12px; font-weight:500;">
                                                ✓ <?php echo esc_html__( 'Ready to boost', '5th-social-bot' ); ?>
                                            </span>
                                        <?php } ?>
                                    <?php endif; ?>
                                <?php else : ?>
                                    <span style="color:#999;"><?php echo esc_html__( 'Waiting for Mastodon URL', '5th-social-bot' ); ?></span>
                                <?php endif; ?>
                            </td>
                            <td>
                                <?php
                                $boost_queue_index = array_search( $boost_post_id, $boost_queue, true );
                                if ( false !== $boost_queue_index ) :
                                    ?>
                                    <div style="display:flex; gap:5px; align-items:center;">
                                        <?php if ( $boost_queue_index > 0 ) : ?>
                                            <form method="post" action="" style="display:inline;">
                                                <?php wp_nonce_field( 'fifth_social_bot_settings' ); ?>
                                                <input type="hidden" name="fifth_social_bot_action" value="move_boost_queue_up" />
                                                <input type="hidden" name="fsb_post_id" value="<?php echo esc_attr( $boost_post_id ); ?>" />
                                                <button type="submit" class="button button-small" title="<?php echo esc_attr__( 'Move up in boost queue', '5th-social-bot' ); ?>">↑</button>
                                            </form>
                                        <?php endif; ?>
                                        <?php if ( $boost_queue_index < count( $boost_queue ) - 1 ) : ?>
                                            <form method="post" action="" style="display:inline;">
                                                <?php wp_nonce_field( 'fifth_social_bot_settings' ); ?>
                                                <input type="hidden" name="fifth_social_bot_action" value="move_boost_queue_down" />
                                                <input type="hidden" name="fsb_post_id" value="<?php echo esc_attr( $boost_post_id ); ?>" />
                                                <button type="submit" class="button button-small" title="<?php echo esc_attr__( 'Move down in boost queue', '5th-social-bot' ); ?>">↓</button>
                                            </form>
                                        <?php endif; ?>
                                        <form method="post" action="" style="display:inline;">
                                            <?php wp_nonce_field( 'fifth_social_bot_settings' ); ?>
                                            <input type="hidden" name="fifth_social_bot_action" value="remove_boost_queue" />
                                            <input type="hidden" name="fsb_post_id" value="<?php echo esc_attr( $boost_post_id ); ?>" />
                                            <button type="submit" class="button button-small" style="color:#d63638;" title="<?php echo esc_attr__( 'Remove from boost queue', '5th-social-bot' ); ?>" onclick="return confirm('<?php echo esc_js( __( 'Are you sure you want to remove this post from the boost queue?', '5th-social-bot' ) ); ?>');">
                                                <?php echo esc_html__( 'Remove', '5th-social-bot' ); ?>
                                            </button>
                                        </form>
                                    </div>
                                <?php endif; ?>
                            </td>
                        </tr>
                        <?php
                        $queue_index++;
                        endforeach;
                        ?>
                    </tbody>
                </table>
            <?php endif; ?>
        </div>
        <?php endif; ?>

        <p style="margin-top: 2em; font-size: 12px; color: #666;">
            <?php
            printf(
                /* translators: %1$s: link to social.5th.ro, %2$s: link to plugin.5th.ro */
                wp_kses_post( __( 'Plugin created for WordPress → Mastodon (%1$s) integration. Official website: %2$s', '5th-social-bot' ) ),
                '<a href="https://social.5th.ro" target="_blank" rel="noopener noreferrer">social.5th.ro</a>',
                '<a href="https://plugin.5th.ro" target="_blank" rel="noopener noreferrer">plugin.5th.ro</a>'
            );
            ?>
            <br />
            <?php
            echo wp_kses_post(
                'Signed: <strong>SC ARC MEDIASOFT SRL</strong> — <a href="https://e-agentie.ro" target="_blank" rel="noopener noreferrer">e-agentie.ro</a>'
            );
            ?>
        </p>
        <?php
    }

    /**
     * Shortcode & Verification tab.
     *
     * @param array $settings Settings.
     */
    protected function render_shortcode_tab( $settings ) {
        $username = isset( $settings['mastodon_username'] ) ? $settings['mastodon_username'] : '';
        $username = ltrim( $username, '@' );
        $profile_url = ! empty( $username ) ? trailingslashit( FIFTH_SOCIAL_BOT_INSTANCE ) . '@' . $username : '';
        ?>
        <div class="fsb-card">
            <h2><?php echo esc_html( __( 'Mastodon Button Shortcode', '5th-social-bot' ) ); ?></h2>
            <p>
                <?php echo esc_html__( 'Display a Mastodon button or link on your site. Compatible with Gutenberg, Elementor, and other page builders.', '5th-social-bot' ); ?>
            </p>

            <h3><?php echo esc_html( __( 'Basic Usage', '5th-social-bot' ) ); ?></h3>
        <p>
            <code>[mastodon_button]</code>
        </p>
        <p class="description">
            <?php echo esc_html__( 'Uses your Mastodon username from settings. Displays as a button with Mastodon icon.', '5th-social-bot' ); ?>
        </p>

        <h3><?php echo esc_html( __( 'Button Styles & Preview', '5th-social-bot' ) ); ?></h3>
        
        <div style="padding:20px; background:#f5f5f5; border:1px solid #ddd; border-radius:4px; margin-bottom:20px;">
            <h4 style="margin-top:0;"><?php echo esc_html__( 'Live Preview', '5th-social-bot' ); ?></h4>
            <?php
            if ( ! empty( $profile_url ) ) {
                echo '<div style="margin-bottom:15px;"><strong>' . esc_html__( 'Buttons:', '5th-social-bot' ) . '</strong><br />';
                echo '<div style="margin-top:10px; line-height:2.5;">';
                echo do_shortcode( '[mastodon_button style="button"]' ) . ' ';
                echo do_shortcode( '[mastodon_button style="modern"]' ) . ' ';
                echo do_shortcode( '[mastodon_button style="minimal"]' ) . ' ';
                echo do_shortcode( '[mastodon_button style="rounded"]' ) . ' ';
                echo do_shortcode( '[mastodon_button style="outline"]' ) . ' ';
                echo do_shortcode( '[mastodon_button style="gradient"]' ) . ' ';
                echo do_shortcode( '[mastodon_button style="pill"]' ) . ' ';
                echo do_shortcode( '[mastodon_button style="white"]' ) . ' ';
                echo do_shortcode( '[mastodon_button style="black"]' ) . ' ';
                echo do_shortcode( '[mastodon_button style="white-outline"]' ) . ' ';
                echo do_shortcode( '[mastodon_button style="black-outline"]' ) . ' ';
                echo do_shortcode( '[mastodon_button style="white-minimal"]' ) . ' ';
                echo do_shortcode( '[mastodon_button style="black-minimal"]' ) . ' ';
                echo do_shortcode( '[mastodon_button style="link"]' );
                echo '</div></div>';
                
                echo '<div style="margin-bottom:15px;"><strong>' . esc_html__( 'Icons:', '5th-social-bot' ) . '</strong><br />';
                echo '<div style="margin-top:10px; line-height:2.5;">';
                echo do_shortcode( '[mastodon_button style="icon" icon_color="default"]' ) . ' ';
                echo do_shortcode( '[mastodon_button style="icon" icon_color="black"]' ) . ' ';
                echo do_shortcode( '[mastodon_button style="icon" icon_color="gray"]' ) . ' ';
                echo do_shortcode( '[mastodon_button style="icon" icon_color="red"]' ) . ' ';
                echo do_shortcode( '[mastodon_button style="icon" icon_color="white"]' ) . ' ';
                echo do_shortcode( '[mastodon_button style="icon" icon_color="green"]' ) . ' ';
                echo do_shortcode( '[mastodon_button style="icon" icon_color="purple"]' );
                echo '</div></div>';
                
                echo '<div style="margin-bottom:15px;"><strong>' . esc_html__( 'Sizes:', '5th-social-bot' ) . '</strong><br />';
                echo '<div style="margin-top:10px; line-height:2.5;">';
                echo do_shortcode( '[mastodon_button size="small"]' ) . ' ';
                echo do_shortcode( '[mastodon_button size="medium"]' ) . ' ';
                echo do_shortcode( '[mastodon_button size="large"]' );
                echo '</div></div>';
            } else {
                echo '<p>' . esc_html__( 'Set your Mastodon username in Settings to see preview.', '5th-social-bot' ) . '</p>';
            }
            ?>
        </div>

        <table class="form-table" role="presentation">
            <tr>
                <th scope="row"><?php echo esc_html( __( 'Available Styles', '5th-social-bot' ) ); ?></th>
                <td>
                    <code>[mastodon_button style="button"]</code> - <?php echo esc_html__( 'Classic button (default)', '5th-social-bot' ); ?><br />
                    <code>[mastodon_button style="modern"]</code> - <?php echo esc_html__( 'Modern gradient with shadow', '5th-social-bot' ); ?><br />
                    <code>[mastodon_button style="minimal"]</code> - <?php echo esc_html__( 'Minimal outline style', '5th-social-bot' ); ?><br />
                    <code>[mastodon_button style="rounded"]</code> - <?php echo esc_html__( 'Rounded corners', '5th-social-bot' ); ?><br />
                    <code>[mastodon_button style="outline"]</code> - <?php echo esc_html__( 'Bold outline style', '5th-social-bot' ); ?><br />
                    <code>[mastodon_button style="gradient"]</code> - <?php echo esc_html__( 'Colorful gradient', '5th-social-bot' ); ?><br />
                    <code>[mastodon_button style="pill"]</code> - <?php echo esc_html__( 'Pill-shaped button', '5th-social-bot' ); ?><br />
                    <code>[mastodon_button style="white"]</code> - <?php echo esc_html__( 'White clean style', '5th-social-bot' ); ?><br />
                    <code>[mastodon_button style="black"]</code> - <?php echo esc_html__( 'Black bold style', '5th-social-bot' ); ?><br />
                    <code>[mastodon_button style="white-outline"]</code> - <?php echo esc_html__( 'White outline style', '5th-social-bot' ); ?><br />
                    <code>[mastodon_button style="black-outline"]</code> - <?php echo esc_html__( 'Black outline style', '5th-social-bot' ); ?><br />
                    <code>[mastodon_button style="white-minimal"]</code> - <?php echo esc_html__( 'White minimal style', '5th-social-bot' ); ?><br />
                    <code>[mastodon_button style="black-minimal"]</code> - <?php echo esc_html__( 'Black minimal style', '5th-social-bot' ); ?><br />
                    <code>[mastodon_button style="link"]</code> - <?php echo esc_html__( 'Simple link style', '5th-social-bot' ); ?><br />
                    <code>[mastodon_button style="icon"]</code> - <?php echo esc_html__( 'Icon only (circular)', '5th-social-bot' ); ?>
                </td>
            </tr>
            <tr>
                <th scope="row"><?php echo esc_html( __( 'Button Sizes', '5th-social-bot' ) ); ?></th>
                <td>
                    <code>[mastodon_button size="small"]</code> - <?php echo esc_html__( 'Small button', '5th-social-bot' ); ?><br />
                    <code>[mastodon_button size="medium"]</code> - <?php echo esc_html__( 'Medium button (default)', '5th-social-bot' ); ?><br />
                    <code>[mastodon_button size="large"]</code> - <?php echo esc_html__( 'Large button', '5th-social-bot' ); ?>
                </td>
            </tr>
            <tr>
                <th scope="row"><?php echo esc_html( __( 'Custom URL', '5th-social-bot' ) ); ?></th>
                <td>
                    <code>[mastodon_button url="https://social.5th.ro/@username"]</code>
                    <p class="description" style="margin-top:5px;">
                        <?php
                        printf(
                            /* translators: %s: link to social.5th.ro */
                            esc_html__( 'Example uses %s - replace with your actual Mastodon profile URL.', '5th-social-bot' ),
                            '<a href="https://social.5th.ro" target="_blank" rel="noopener noreferrer">social.5th.ro</a>'
                        );
                        ?>
                    </p>
                </td>
            </tr>
            <tr>
                <th scope="row"><?php echo esc_html( __( 'Custom Text', '5th-social-bot' ) ); ?></th>
                <td>
                    <code>[mastodon_button text="Follow me on Mastodon"]</code>
                </td>
            </tr>
            <tr>
                <th scope="row"><?php echo esc_html( __( 'Icon Options', '5th-social-bot' ) ); ?></th>
                <td>
                    <code>[mastodon_button icon="yes"]</code> - <?php echo esc_html__( 'Show icon (default)', '5th-social-bot' ); ?><br />
                    <code>[mastodon_button icon="no"]</code> - <?php echo esc_html__( 'Hide icon', '5th-social-bot' ); ?><br />
                    <code>[mastodon_button style="icon" icon_color="black"]</code> - <?php echo esc_html__( 'Icon only with black color', '5th-social-bot' ); ?><br />
                    <code>[mastodon_button style="icon" icon_color="gray"]</code> - <?php echo esc_html__( 'Icon only with gray color', '5th-social-bot' ); ?><br />
                    <code>[mastodon_button style="icon" icon_color="red"]</code> - <?php echo esc_html__( 'Icon only with red color', '5th-social-bot' ); ?><br />
                    <p class="description">
                        <?php echo esc_html__( 'Available icon colors (for icon style only): default (blue), black, gray, red, white, green, purple', '5th-social-bot' ); ?>
                    </p>
                </td>
            </tr>
            <tr>
                <th scope="row"><?php echo esc_html( __( 'Custom CSS Class', '5th-social-bot' ) ); ?></th>
                <td>
                    <code>[mastodon_button class="my-custom-class"]</code>
                </td>
            </tr>
        </table>
        
        <p class="description">
            <strong><?php echo esc_html__( 'Mobile Friendly:', '5th-social-bot' ); ?></strong>
            <?php echo esc_html__( 'All button styles are fully responsive and mobile-friendly. On mobile devices, buttons automatically adjust their size and layout for optimal viewing.', '5th-social-bot' ); ?>
        </p>

        </div>

        <div class="fsb-card">
            <h2><?php echo esc_html( __( 'Mastodon Feed Shortcode', '5th-social-bot' ) ); ?></h2>
            <p>
                <?php echo esc_html__( 'Display your Mastodon feed or tag feed on your WordPress site. Perfect for sidebars, widgets, or pages.', '5th-social-bot' ); ?>
            </p>

        <h3><?php echo esc_html( __( 'User Feed (Automatic)', '5th-social-bot' ) ); ?></h3>
        <p>
            <code>[mastodon_feed]</code>
        </p>
        <p class="description">
            <?php echo esc_html__( 'Automatically uses your Mastodon username from settings (the username associated with your access token). Displays your latest posts.', '5th-social-bot' ); ?>
            <?php if ( ! empty( $settings['mastodon_username'] ) ) : ?>
                <br />
                <strong><?php echo esc_html__( 'Current username:', '5th-social-bot' ); ?></strong> 
                <code>@<?php echo esc_html( $settings['mastodon_username'] ); ?></code>
            <?php else : ?>
                <br />
                <span style="color:#d98500;">⚠ <?php echo esc_html__( 'No username set. Set your Mastodon username in Settings tab or it will be automatically retrieved when you save your access token.', '5th-social-bot' ); ?></span>
            <?php endif; ?>
        </p>

        <h3><?php echo esc_html( __( 'Custom User Feed', '5th-social-bot' ) ); ?></h3>
        <p>
            <code>[mastodon_feed username="username" count="10"]</code>
        </p>
        <p class="description">
            <?php echo esc_html__( 'Display feed from a specific user (overrides the automatic username from settings). Count: number of posts (1-20, default: 5).', '5th-social-bot' ); ?>
        </p>

        <h3><?php echo esc_html( __( 'Tag Feed', '5th-social-bot' ) ); ?></h3>
        <p>
            <code>[mastodon_feed tag="wordpress" count="5"]</code>
        </p>
        <p class="description">
            <?php echo esc_html__( 'Display posts with a specific hashtag. Count: number of posts (1-20, default: 5).', '5th-social-bot' ); ?>
        </p>

        <h3><?php echo esc_html( __( 'Advanced Options', '5th-social-bot' ) ); ?></h3>
        <table class="form-table" role="presentation">
            <tr>
                <th scope="row"><?php echo esc_html( __( 'Cache Time', '5th-social-bot' ) ); ?></th>
                <td>
                    <code>[mastodon_feed cache_time="600"]</code>
                    <p class="description">
                        <?php echo esc_html__( 'Cache time in seconds (default: 300 = 5 minutes).', '5th-social-bot' ); ?>
                    </p>
                </td>
            </tr>
            <tr>
                <th scope="row"><?php echo esc_html( __( 'Custom Instance', '5th-social-bot' ) ); ?></th>
                <td>
                    <code>[mastodon_feed instance="https://social.5th.ro"]</code>
                    <p class="description">
                        <?php
                        printf(
                            /* translators: %s: link to social.5th.ro */
                            esc_html__( 'Use a different Mastodon instance (default: %s). Automatically corrects common typos like social5th.ro → social.5th.ro', '5th-social-bot' ),
                            '<a href="https://social.5th.ro" target="_blank" rel="noopener noreferrer">social.5th.ro</a>'
                        );
                        ?>
                    </p>
                </td>
            </tr>
            <tr>
                <th scope="row"><?php echo esc_html( __( 'Exclude Replies', '5th-social-bot' ) ); ?></th>
                <td>
                    <code>[mastodon_feed exclude_replies="true"]</code>
                    <p class="description">
                        <?php echo esc_html__( 'Exclude reply posts from the feed (default: false).', '5th-social-bot' ); ?>
                    </p>
                </td>
            </tr>
            <tr>
                <th scope="row"><?php echo esc_html( __( 'Exclude Reblogs', '5th-social-bot' ) ); ?></th>
                <td>
                    <code>[mastodon_feed exclude_reblogs="true"]</code>
                    <p class="description">
                        <?php echo esc_html__( 'Exclude reblogged posts from the feed (default: false).', '5th-social-bot' ); ?>
                    </p>
                </td>
            </tr>
            <tr>
                <th scope="row"><?php echo esc_html( __( 'Only Media Posts', '5th-social-bot' ) ); ?></th>
                <td>
                    <code>[mastodon_feed only_media="true"]</code>
                    <p class="description">
                        <?php echo esc_html__( 'Show only posts that contain media (images, videos) (default: false).', '5th-social-bot' ); ?>
                    </p>
                </td>
            </tr>
            <tr>
                <th scope="row"><?php echo esc_html( __( 'Layout Style', '5th-social-bot' ) ); ?></th>
                <td>
                    <code>[mastodon_feed layout="default"]</code> - <?php echo esc_html__( 'Default layout (default)', '5th-social-bot' ); ?><br />
                    <code>[mastodon_feed layout="compact"]</code> - <?php echo esc_html__( 'Compact layout for sidebars', '5th-social-bot' ); ?><br />
                    <code>[mastodon_feed layout="grid"]</code> - <?php echo esc_html__( 'Grid layout (responsive columns)', '5th-social-bot' ); ?>
                </td>
            </tr>
            <tr>
                <th scope="row"><?php echo esc_html( __( 'Custom CSS Class', '5th-social-bot' ) ); ?></th>
                <td>
                    <code>[mastodon_feed custom_css="my-custom-class"]</code>
                    <p class="description">
                        <?php echo esc_html__( 'Add a custom CSS class to the feed container for additional styling.', '5th-social-bot' ); ?>
                    </p>
                </td>
            </tr>
            </table>
        </div>

        <div class="fsb-card">
            <h2><?php echo esc_html( __( 'Examples', '5th-social-bot' ) ); ?></h2>
            <div class="fsb-code-block">
                <ul style="margin:0; padding-left:20px;">
                    <li><code>[mastodon_feed]</code> - <?php echo esc_html__( 'Your feed (automatically uses username from settings, 5 posts)', '5th-social-bot' ); ?></li>
                    <li><code>[mastodon_feed count="10"]</code> - <?php echo esc_html__( 'Your feed (automatically uses username from settings, 10 posts)', '5th-social-bot' ); ?></li>
                    <li><code>[mastodon_feed tag="wordpress"]</code> - <?php echo esc_html__( 'WordPress tag feed', '5th-social-bot' ); ?></li>
                    <li><code>[mastodon_feed username="otheruser" count="3"]</code> - <?php echo esc_html__( 'Another user\'s feed (overrides automatic username)', '5th-social-bot' ); ?></li>
                    <li><code>[mastodon_feed exclude_replies="true" exclude_reblogs="true"]</code> - <?php echo esc_html__( 'Only original posts, no replies or reblogs', '5th-social-bot' ); ?></li>
                    <li><code>[mastodon_feed only_media="true" count="5"]</code> - <?php echo esc_html__( 'Only posts with images/videos', '5th-social-bot' ); ?></li>
                    <li><code>[mastodon_feed layout="compact" count="3"]</code> - <?php echo esc_html__( 'Compact layout for sidebars', '5th-social-bot' ); ?></li>
                    <li><code>[mastodon_feed layout="grid" count="6"]</code> - <?php echo esc_html__( 'Grid layout with 6 posts', '5th-social-bot' ); ?></li>
                    <li><code>[mastodon_feed custom_css="my-feed"]</code> - <?php echo esc_html__( 'With custom CSS class', '5th-social-bot' ); ?></li>
                </ul>
            </div>
            
            <div class="fsb-info-box">
                <p style="margin:0;">
                    <strong><?php echo esc_html__( 'Note:', '5th-social-bot' ); ?></strong>
                    <?php echo esc_html__( 'Maximum 10 posts can be displayed in the feed. If you specify more than 10, only 10 will be shown.', '5th-social-bot' ); ?>
                </p>
            </div>
        </div>

        <div class="fsb-card">
            <h2><?php echo esc_html( __( 'Widget Support', '5th-social-bot' ) ); ?></h2>
        <p>
            <?php echo esc_html__( 'Both shortcodes work in WordPress widgets. Add them to your sidebar or footer using the Text widget or Custom HTML widget.', '5th-social-bot' ); ?>
        </p>
            <p>
                <?php echo esc_html__( 'For Gutenberg, use the Shortcode block. For Elementor, use the Shortcode widget.', '5th-social-bot' ); ?>
            </p>
        </div>
        <p style="margin-top: 2em; font-size: 12px; color: #666;">
            <?php
            printf(
                /* translators: %1$s: link to social.5th.ro, %2$s: link to plugin.5th.ro */
                wp_kses_post( __( 'Plugin created for WordPress → Mastodon (%1$s) integration. Official website: %2$s', '5th-social-bot' ) ),
                '<a href="https://social.5th.ro" target="_blank" rel="noopener noreferrer">social.5th.ro</a>',
                '<a href="https://plugin.5th.ro" target="_blank" rel="noopener noreferrer">plugin.5th.ro</a>'
            );
            ?>
            <br />
            <?php
            echo wp_kses_post(
                'Signed: <strong>SC ARC MEDIASOFT SRL</strong> — <a href="https://e-agentie.ro" target="_blank" rel="noopener noreferrer">e-agentie.ro</a>'
            );
            ?>
        </p>
        <?php
    }

    /**
     * Render Analytics tab.
     */
    protected function render_analytics_tab() {
        $aggregated = Fifth_Social_Bot_Analytics::get_aggregated_stats();
        $all_posts = Fifth_Social_Bot_Analytics::get_all_posts_stats( 50 );
        ?>
        <div class="fsb-card">
            <h2><?php echo esc_html( __( 'Mastodon Post Analytics', '5th-social-bot' ) ); ?></h2>
            <p class="description">
                <?php echo esc_html__( 'Statistics for your posts on Mastodon. Data is cached for 5 minutes to reduce API calls.', '5th-social-bot' ); ?>
            </p>
        </div>

        <div class="fsb-card">
            <h2><?php echo esc_html( __( 'Aggregated Statistics', '5th-social-bot' ) ); ?></h2>
            
            <div class="fsb-stats-grid">
                <div class="fsb-stat-card">
                    <div class="stat-value"><?php echo absint( $aggregated['total_posts'] ); ?></div>
                    <div class="stat-label"><?php echo esc_html__( 'Total Posts', '5th-social-bot' ); ?></div>
                </div>
                <div class="fsb-stat-card">
                    <div class="stat-value"><?php echo absint( $aggregated['total_replies'] ); ?></div>
                    <div class="stat-label"><?php echo esc_html__( 'Total Replies', '5th-social-bot' ); ?></div>
                </div>
                <div class="fsb-stat-card">
                    <div class="stat-value"><?php echo absint( $aggregated['total_reblogs'] ); ?></div>
                    <div class="stat-label"><?php echo esc_html__( 'Total Boosts', '5th-social-bot' ); ?></div>
                </div>
                <div class="fsb-stat-card">
                    <div class="stat-value"><?php echo absint( $aggregated['total_favourites'] ); ?></div>
                    <div class="stat-label"><?php echo esc_html__( 'Total Favorites', '5th-social-bot' ); ?></div>
                </div>
                <?php if ( $aggregated['total_views'] > 0 ) : ?>
                <div class="fsb-stat-card">
                    <div class="stat-value"><?php echo absint( $aggregated['total_views'] ); ?></div>
                    <div class="stat-label"><?php echo esc_html__( 'Total Views', '5th-social-bot' ); ?></div>
                </div>
                <?php endif; ?>
            </div>

            <div class="fsb-stats-grid" style="margin-top: 20px;">
                <div class="fsb-stat-card">
                    <div class="stat-value"><?php echo esc_html( number_format( $aggregated['avg_replies'], 1 ) ); ?></div>
                    <div class="stat-label"><?php echo esc_html__( 'Avg Replies/Post', '5th-social-bot' ); ?></div>
                </div>
                <div class="fsb-stat-card">
                    <div class="stat-value"><?php echo esc_html( number_format( $aggregated['avg_reblogs'], 1 ) ); ?></div>
                    <div class="stat-label"><?php echo esc_html__( 'Avg Boosts/Post', '5th-social-bot' ); ?></div>
                </div>
                <div class="fsb-stat-card">
                    <div class="stat-value"><?php echo esc_html( number_format( $aggregated['avg_favourites'], 1 ) ); ?></div>
                    <div class="stat-label"><?php echo esc_html__( 'Avg Favorites/Post', '5th-social-bot' ); ?></div>
                </div>
                <?php if ( $aggregated['total_views'] > 0 ) : ?>
                <div class="fsb-stat-card">
                    <div class="stat-value"><?php echo esc_html( number_format( $aggregated['avg_views'], 1 ) ); ?></div>
                    <div class="stat-label"><?php echo esc_html__( 'Avg Views/Post', '5th-social-bot' ); ?></div>
                </div>
                <?php endif; ?>
            </div>

            <?php if ( $aggregated['posts_with_boosts'] > 0 ) : ?>
            <div class="fsb-card" style="margin-top: 20px;">
                <h3><?php echo esc_html__( 'Boost Statistics', '5th-social-bot' ); ?></h3>
                <div class="fsb-stats-grid">
                    <div class="fsb-stat-card">
                        <div class="stat-value"><?php echo absint( $aggregated['posts_with_boosts'] ); ?></div>
                        <div class="stat-label"><?php echo esc_html__( 'Posts with Boosts', '5th-social-bot' ); ?></div>
                    </div>
                    <div class="fsb-stat-card">
                        <div class="stat-value"><?php echo absint( $aggregated['total_boost_replies'] ); ?></div>
                        <div class="stat-label"><?php echo esc_html__( 'Boost Replies', '5th-social-bot' ); ?></div>
                    </div>
                    <div class="fsb-stat-card">
                        <div class="stat-value"><?php echo absint( $aggregated['total_boost_reblogs'] ); ?></div>
                        <div class="stat-label"><?php echo esc_html__( 'Boost Boosts', '5th-social-bot' ); ?></div>
                    </div>
                    <div class="fsb-stat-card">
                        <div class="stat-value"><?php echo absint( $aggregated['total_boost_favourites'] ); ?></div>
                        <div class="stat-label"><?php echo esc_html__( 'Boost Favorites', '5th-social-bot' ); ?></div>
                    </div>
                </div>
            </div>
            <?php endif; ?>
        </div>

        <div class="fsb-card">
            <h2><?php echo esc_html__( 'Post Statistics', '5th-social-bot' ); ?></h2>
            <p class="description">
                <?php echo esc_html__( 'Individual statistics for each post. Showing up to 50 most recent posts.', '5th-social-bot' ); ?>
            </p>
            
            <form method="post" action="" style="margin-bottom:1em;">
                <?php wp_nonce_field( 'fifth_social_bot_settings' ); ?>
                <input type="hidden" name="fifth_social_bot_action" value="import_stats" />
                <button type="submit" class="button button-secondary">
                    <?php echo esc_html__( 'Import Stats from Mastodon', '5th-social-bot' ); ?>
                </button>
                <p class="description" style="margin-top: 0.5em;">
                    <?php echo esc_html__( 'Import all available posts from your Mastodon account. This will update missing Mastodon URLs for posts that were posted but the URL was not saved.', '5th-social-bot' ); ?>
                </p>
            </form>
            
            <?php if ( empty( $all_posts ) ) : ?>
                <div class="fsb-info-box">
                    <p style="margin: 0;">
                        <strong><?php echo esc_html__( 'No posts found', '5th-social-bot' ); ?></strong>
                        <br />
                        <?php echo esc_html__( 'No posts have been posted to Mastodon yet. Posts will appear here after they are successfully posted.', '5th-social-bot' ); ?>
                    </p>
                </div>
            <?php else : ?>
                <table class="widefat fixed striped">
                    <thead>
                        <tr>
                            <th><?php echo esc_html__( 'Post', '5th-social-bot' ); ?></th>
                            <th><?php echo esc_html__( 'Date', '5th-social-bot' ); ?></th>
                            <th><?php echo esc_html__( 'Replies', '5th-social-bot' ); ?></th>
                            <th><?php echo esc_html__( 'Boosts', '5th-social-bot' ); ?></th>
                            <th><?php echo esc_html__( 'Favorites', '5th-social-bot' ); ?></th>
                            <?php if ( $aggregated['total_views'] > 0 ) : ?>
                            <th><?php echo esc_html__( 'Views', '5th-social-bot' ); ?></th>
                            <?php endif; ?>
                            <th><?php echo esc_html__( 'Actions', '5th-social-bot' ); ?></th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ( $all_posts as $post_data ) : ?>
                            <tr>
                                <td>
                                    <strong><?php echo esc_html( $post_data['post_title'] ); ?></strong>
                                    <br />
                                    <small>ID: <?php echo absint( $post_data['post_id'] ); ?></small>
                                </td>
                                <td><?php echo esc_html( gmdate( 'Y-m-d H:i', strtotime( $post_data['post_date'] ) ) ); ?></td>
                                <td><?php echo absint( $post_data['stats']['replies_count'] ); ?></td>
                                <td><?php echo absint( $post_data['stats']['reblogs_count'] ); ?></td>
                                <td><?php echo absint( $post_data['stats']['favourites_count'] ); ?></td>
                                <?php if ( $aggregated['total_views'] > 0 ) : ?>
                                <td><?php echo absint( $post_data['stats']['views_count'] ); ?></td>
                                <?php endif; ?>
                                <td>
                                    <?php if ( ! empty( $post_data['mastodon_url'] ) ) : ?>
                                        <a href="<?php echo esc_url( $post_data['mastodon_url'] ); ?>" target="_blank" rel="noopener noreferrer" class="button button-small">
                                            <?php echo esc_html__( 'View Post', '5th-social-bot' ); ?>
                                        </a>
                                    <?php endif; ?>
                                    <?php if ( ! empty( $post_data['boost_url'] ) ) : ?>
                                        <br />
                                        <a href="<?php echo esc_url( $post_data['boost_url'] ); ?>" target="_blank" rel="noopener noreferrer" class="button button-small" style="margin-top: 5px;">
                                            <?php echo esc_html__( 'View Boost', '5th-social-bot' ); ?>
                                        </a>
                                    <?php endif; ?>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            <?php endif; ?>
        </div>
        
        <p style="margin-top: 2em; font-size: 12px; color: #666;">
            <?php
            printf(
                /* translators: %1$s: link to social.5th.ro, %2$s: link to plugin.5th.ro */
                wp_kses_post( __( 'Plugin created for WordPress → Mastodon (%1$s) integration. Official website: %2$s', '5th-social-bot' ) ),
                '<a href="https://social.5th.ro" target="_blank" rel="noopener noreferrer">social.5th.ro</a>',
                '<a href="https://plugin.5th.ro" target="_blank" rel="noopener noreferrer">plugin.5th.ro</a>'
            );
            ?>
            <br />
            <?php
            echo wp_kses_post(
                'Signed: <strong>SC ARC MEDIASOFT SRL</strong> — <a href="https://e-agentie.ro" target="_blank" rel="noopener noreferrer">e-agentie.ro</a>'
            );
            ?>
        </p>
        <?php
    }

    /**
     * Log & debug tab.
     */
    protected function render_log_tab() {
        $log = Fifth_Social_Bot_Logger::get_log();
        $settings = Fifth_Social_Bot_Core::get_settings();
        $debug_mode = ! empty( $settings['debug_mode'] );
        ?>
        <div class="fsb-card">
            <h2><?php echo esc_html( __( 'Log & debug', '5th-social-bot' ) ); ?></h2>
            <p>
                <?php echo esc_html( __( 'Here you can see the latest events related to Mastodon posting: errors, successful posts, queue, etc. By default only the last 20 entries are shown. You can progressively load more.', '5th-social-bot' ) ); ?>
            </p>
            
            <?php if ( $debug_mode ) : ?>
                <div class="fsb-info-box success">
                <p>
                    <strong><?php echo esc_html__( 'Debug Mode: ON', '5th-social-bot' ); ?></strong>
                    <br />
                    <?php echo esc_html__( 'Debug mode is enabled. You will see additional detailed information in the log, including:', '5th-social-bot' ); ?>
                    <ul style="margin-left: 20px; margin-top: 10px;">
                        <li><?php echo esc_html__( 'Shortcode calls with parameters', '5th-social-bot' ); ?></li>
                        <li><?php echo esc_html__( 'API requests and responses', '5th-social-bot' ); ?></li>
                        <li><?php echo esc_html__( 'Cache hits/misses for feeds', '5th-social-bot' ); ?></li>
                        <li><?php echo esc_html__( 'Detailed error information', '5th-social-bot' ); ?></li>
                        <li><?php echo esc_html__( 'Post composition details', '5th-social-bot' ); ?></li>
                    </ul>
                    <?php echo esc_html__( 'Look for entries prefixed with [Debug] or [Shortcode] in the log below.', '5th-social-bot' ); ?>
                </p>
                </div>
            <?php else : ?>
                <div class="fsb-info-box warning">
                <p>
                    <strong><?php echo esc_html__( 'Debug Mode: OFF', '5th-social-bot' ); ?></strong>
                    <br />
                    <?php echo esc_html__( 'Debug mode is disabled. Only basic error and success messages are logged.', '5th-social-bot' ); ?>
                    <br />
                    <?php echo esc_html__( 'To enable detailed logging, go to Settings and enable "Debug Mode".', '5th-social-bot' ); ?>
                </p>
            </div>
        <?php endif; ?>

            <form method="post" action="" style="margin-bottom:1em;">
                <?php wp_nonce_field( 'fifth_social_bot_settings' ); ?>
                <input type="hidden" name="fifth_social_bot_action" value="clear_log" />
                <button type="submit" class="button">
                    <?php echo esc_html( __( 'Clear log', '5th-social-bot' ) ); ?>
                </button>
            </form>
        </div>

        <div class="fsb-card">
            <h2><?php echo esc_html( __( 'Log Entries', '5th-social-bot' ) ); ?></h2>
            
            <style>
                .fsb-log-hidden { display:none; }
            </style>

            <table class="widefat fixed striped">
            <thead>
                <tr>
                    <th><?php echo esc_html( __( 'Date', '5th-social-bot' ) ); ?></th>
                    <th><?php echo esc_html( __( 'Level', '5th-social-bot' ) ); ?></th>
                    <th><?php echo esc_html( __( 'Message', '5th-social-bot' ) ); ?></th>
                </tr>
            </thead>
            <tbody>
            <?php if ( empty( $log ) ) : ?>
                <tr><td colspan="3"><?php echo esc_html( __( 'Log is empty.', '5th-social-bot' ) ); ?></td></tr>
            <?php else : ?>
                <?php
                $entries = array_reverse( $log );
                $count   = 0;
                foreach ( $entries as $entry ) :
                    $row_class = ( $count >= 20 ) ? 'fsb-log-row fsb-log-hidden' : 'fsb-log-row';
                    ?>
                    <tr class="<?php echo esc_attr( $row_class ); ?>">
                        <td><?php echo esc_html( isset( $entry['time'] ) ? $entry['time'] : '' ); ?></td>
                        <td><?php echo esc_html( isset( $entry['level'] ) ? $entry['level'] : '' ); ?></td>
                        <td><?php echo esc_html( isset( $entry['message'] ) ? $entry['message'] : '' ); ?></td>
                    </tr>
                    <?php
                    $count++;
                endforeach;
            endif;
            ?>
            </tbody>
        </table>

        <?php if ( ! empty( $log ) && count( $log ) > 20 ) : ?>
            <p style="margin-top:1em;">
                <button type="button" class="button" id="fsb-load-more-log">
                    <?php echo esc_html( __( 'Load 20 more', '5th-social-bot' ) ); ?>
                </button>
            </p>
            <script>
            (function(){
                var btn = document.getElementById('fsb-load-more-log');
                if (!btn) return;
                btn.addEventListener('click', function(e){
                    e.preventDefault();
                    var hidden = document.querySelectorAll('.fsb-log-row.fsb-log-hidden');
                    var perPage = 20;
                    var shown = 0;
                    for (var i = 0; i < hidden.length && shown < perPage; i++) {
                        hidden[i].classList.remove('fsb-log-hidden');
                        shown++;
                    }
                    if (document.querySelectorAll('.fsb-log-row.fsb-log-hidden').length === 0) {
                        btn.style.display = 'none';
                    }
                });
            })();
            </script>
            <?php endif; ?>
        </div>
        <p style="margin-top: 2em; font-size: 12px; color: #666;">
            <?php
            printf(
                /* translators: %1$s: link to social.5th.ro, %2$s: link to plugin.5th.ro */
                wp_kses_post( __( 'Plugin created for WordPress → Mastodon (%1$s) integration. Official website: %2$s', '5th-social-bot' ) ),
                '<a href="https://social.5th.ro" target="_blank" rel="noopener noreferrer">social.5th.ro</a>',
                '<a href="https://plugin.5th.ro" target="_blank" rel="noopener noreferrer">plugin.5th.ro</a>'
            );
            ?>
            <br />
            <?php
            echo wp_kses_post(
                'Signed: <strong>SC ARC MEDIASOFT SRL</strong> — <a href="https://e-agentie.ro" target="_blank" rel="noopener noreferrer">e-agentie.ro</a>'
            );
            ?>
        </p>
        <?php
    }

    /**
     * Auto-Boost tab.
     *
     * @param array  $settings Settings.
     * @param string $boost_connection_message Connection test message.
     */
    protected function render_boost_tab( $settings, $boost_connection_message = '' ) {
        
        // Check if primary account is configured (required).
        $has_primary_account = ! empty( $settings['access_token'] );
        ?>
        <div class="fsb-card">
            <h2><?php echo esc_html( __( 'Auto-Boost Settings', '5th-social-bot' ) ); ?></h2>
            <p>
                <?php
                printf(
                    /* translators: %s: link to social.5th.ro */
                    esc_html__( 'Configure a second Mastodon instance to automatically boost your posts after they are published on %s.', '5th-social-bot' ),
                    '<a href="https://social.5th.ro" target="_blank" rel="noopener noreferrer">social.5th.ro</a>'
                );
                ?>
            </p>
            
            <?php if ( ! $has_primary_account ) : ?>
                <div class="fsb-info-box warning">
                <p>
                    <strong><?php echo esc_html__( 'Primary account required:', '5th-social-bot' ); ?></strong>
                    <?php
                    printf(
                        /* translators: %s: link to social.5th.ro */
                        esc_html__( 'You must configure your primary Mastodon account (%s) in the Settings tab before you can set up auto-boost.', '5th-social-bot' ),
                        '<a href="https://social.5th.ro" target="_blank" rel="noopener noreferrer">social.5th.ro</a>'
                    );
                    ?>
                </p>
                </div>
            <?php endif; ?>
        </div>
        
        <form method="post" action="">
            <?php wp_nonce_field( 'fifth_social_bot_settings' ); ?>
            <input type="hidden" name="fifth_social_bot_action" value="save_settings" />

            <div class="fsb-card">
                <h2><?php echo esc_html( __( 'Boost Configuration', '5th-social-bot' ) ); ?></h2>
                <table class="form-table" role="presentation">
                <tr>
                    <th scope="row"><?php echo esc_html( __( 'Enable Auto-Boost', '5th-social-bot' ) ); ?></th>
                    <td>
                        <label>
                            <input type="checkbox" name="fsb_boost_enabled" value="1" <?php checked( isset( $settings['boost_enabled'] ) ? $settings['boost_enabled'] : 0, 1 ); ?> <?php disabled( ! $has_primary_account ); ?> />
                            <?php
                            printf(
                                /* translators: %s: link to social.5th.ro */
                                esc_html__( 'Automatically boost posts on second instance after posting to %s', '5th-social-bot' ),
                                '<a href="https://social.5th.ro" target="_blank" rel="noopener noreferrer">social.5th.ro</a>'
                            );
                            ?>
                        </label>
                        <p class="description">
                            <?php
                            printf(
                                /* translators: %s: link to social.5th.ro */
                                esc_html__( 'When enabled, after a post is successfully published on %s, it will be automatically boosted (reblogged) on the second Mastodon instance configured below.', '5th-social-bot' ),
                                '<a href="https://social.5th.ro" target="_blank" rel="noopener noreferrer">social.5th.ro</a>'
                            );
                            ?>
                        </p>
                    </td>
                </tr>
                
                <tr>
                    <th scope="row"><label for="fsb_boost_instance"><?php echo esc_html( __( 'Second Instance URL', '5th-social-bot' ) ); ?></label></th>
                    <td>
                        <input type="text" id="fsb_boost_instance" name="fsb_boost_instance" class="regular-text" value="<?php echo esc_attr( isset( $settings['boost_instance'] ) ? $settings['boost_instance'] : '' ); ?>" placeholder="https://mastodon.social" <?php disabled( ! $has_primary_account ); ?> />
                        <p class="description">
                            <?php
                            printf(
                                /* translators: %s: link to social.5th.ro */
                                esc_html__( 'URL of the second Mastodon instance (e.g., https://mastodon.social, https://mstdn.social). This instance will boost your posts from %s.', '5th-social-bot' ),
                                '<a href="https://social.5th.ro" target="_blank" rel="noopener noreferrer">social.5th.ro</a>'
                            );
                            ?>
                        </p>
                    </td>
                </tr>
                
                <tr>
                    <th scope="row"><label for="fsb_boost_access_token"><?php echo esc_html( __( 'Second Instance Access Token', '5th-social-bot' ) ); ?></label></th>
                    <td>
                        <div style="position: relative; display: inline-block; width: 100%; max-width: 500px;">
                            <input type="password" id="fsb_boost_access_token" name="fsb_boost_access_token" class="regular-text" value="<?php echo esc_attr( isset( $settings['boost_access_token'] ) ? $settings['boost_access_token'] : '' ); ?>" autocomplete="off" style="padding-right: 50px; width: 100%;" <?php disabled( ! $has_primary_account ); ?> />
                            <button type="button" id="fsb-toggle-boost-token" class="button" style="position: absolute; right: 0; top: 0; height: 30px; padding: 0 10px; border-left: 1px solid #ddd; background: #f5f5f5; cursor: pointer;" title="<?php echo esc_attr__( 'Show/Hide token', '5th-social-bot' ); ?>" <?php disabled( ! $has_primary_account ); ?>>
                                <span id="fsb-toggle-boost-token-text"><?php echo esc_html__( 'Show', '5th-social-bot' ); ?></span>
                            </button>
                        </div>
                        <p class="description">
                            <?php echo esc_html__( 'Personal access token from your account on the second Mastodon instance. Required permissions: read:statuses (to read posts) and write:statuses (to boost/reblog).', '5th-social-bot' ); ?>
                            <br />
                            <strong><?php echo esc_html__( 'Security:', '5th-social-bot' ); ?></strong>
                            <?php echo esc_html__( 'The token is hidden by default for security. Click "Show" to reveal it when needed.', '5th-social-bot' ); ?>
                        </p>
                    </td>
                </tr>
                
                <tr>
                    <th scope="row"><label for="fsb_boost_username"><?php echo esc_html( __( 'Second Instance Username', '5th-social-bot' ) ); ?></label></th>
                    <td>
                        <?php
                        $boost_username = isset( $settings['boost_username'] ) ? $settings['boost_username'] : '';
                        if ( ! empty( $boost_username ) ) :
                            ?>
                            <input type="text" id="fsb_boost_username" class="regular-text" value="<?php echo esc_attr( '@' . ltrim( $boost_username, '@' ) ); ?>" readonly="readonly" style="background-color:#f5f5f5; cursor:not-allowed;" />
                            <p class="description">
                                <?php echo esc_html__( 'Username is automatically retrieved from your Second Instance Access Token and cannot be edited manually.', '5th-social-bot' ); ?>
                            </p>
                        <?php else : ?>
                            <input type="text" id="fsb_boost_username" class="regular-text" value="" readonly="readonly" style="background-color:#f5f5f5; cursor:not-allowed;" placeholder="<?php echo esc_attr__( 'Will be set automatically when you save your Access Token', '5th-social-bot' ); ?>" />
                            <p class="description">
                                <?php echo esc_html__( 'Username will be automatically retrieved from your Second Instance Access Token when you save it.', '5th-social-bot' ); ?>
                            </p>
                        <?php endif; ?>
                    </td>
                </tr>

                <tr>
                    <th scope="row"><label for="fsb_boost_max_per_day"><?php echo esc_html( __( 'Maximum boosts per day', '5th-social-bot' ) ); ?></label></th>
                    <td>
                        <input type="number" id="fsb_boost_max_per_day" name="fsb_boost_max_per_day" min="1" max="10" step="1" value="<?php echo esc_attr( isset( $settings['boost_max_per_day'] ) ? (int) $settings['boost_max_per_day'] : 10 ); ?>" class="small-text" <?php disabled( ! $has_primary_account ); ?> />
                        <p class="description">
                            <?php echo esc_html__( 'Maximum number of boosts allowed per day (1-10). Default: 10.', '5th-social-bot' ); ?>
                        </p>
                    </td>
                </tr>

                <tr>
                    <th scope="row"><label for="fsb_boost_min_interval"><?php echo esc_html( __( 'Minimum interval between boosts', '5th-social-bot' ) ); ?></label></th>
                    <td>
                        <input type="number" id="fsb_boost_min_interval" name="fsb_boost_min_interval" min="30" max="120" step="1" value="<?php echo esc_attr( isset( $settings['boost_min_interval'] ) ? (int) $settings['boost_min_interval'] : 30 ); ?>" class="small-text" <?php disabled( ! $has_primary_account ); ?> />
                        <span style="margin-left:5px;"><?php echo esc_html__( 'minutes', '5th-social-bot' ); ?></span>
                        <p class="description">
                            <?php echo esc_html__( 'Minimum time interval between boosts in minutes (30-120 minutes, i.e., 30 min to 2 hours). Default: 30 minutes.', '5th-social-bot' ); ?>
                        </p>
                    </td>
                </tr>
                </table>

                <p class="submit">
                    <button type="submit" class="button button-primary" <?php disabled( ! $has_primary_account ); ?>>
                        <?php echo esc_html( __( 'Save settings', '5th-social-bot' ) ); ?>
                    </button>
                </p>
            </div>
        </form>

        <div class="fsb-card">
            <h2><?php echo esc_html( __( 'Connection Test', '5th-social-bot' ) ); ?></h2>
        <form method="post" action="" style="margin-bottom:1em;">
            <?php wp_nonce_field( 'fifth_social_bot_settings' ); ?>
            <input type="hidden" name="fifth_social_bot_action" value="check_boost_connection" />
            <table class="form-table" role="presentation">
                <tr>
                    <th scope="row"><label for="fsb_test_boost_instance"><?php echo esc_html( __( 'Instance URL', '5th-social-bot' ) ); ?></label></th>
                    <td>
                        <input type="text" id="fsb_test_boost_instance" name="fsb_boost_instance" class="regular-text" value="<?php echo esc_attr( isset( $settings['boost_instance'] ) ? $settings['boost_instance'] : '' ); ?>" placeholder="https://mastodon.social" <?php disabled( ! $has_primary_account ); ?> />
                    </td>
                </tr>
                <tr>
                    <th scope="row"><label for="fsb_test_boost_token"><?php echo esc_html( __( 'Access Token', '5th-social-bot' ) ); ?></label></th>
                    <td>
                        <input type="password" id="fsb_test_boost_token" name="fsb_boost_access_token" class="regular-text" value="<?php echo esc_attr( isset( $settings['boost_access_token'] ) ? $settings['boost_access_token'] : '' ); ?>" <?php disabled( ! $has_primary_account ); ?> />
                    </td>
                </tr>
            </table>
            <p>
                <button type="submit" class="button" <?php disabled( ! $has_primary_account ); ?>>
                    <?php echo esc_html( __( 'Test connection with second instance', '5th-social-bot' ) ); ?>
                </button>
            </p>
        </form>

            <?php if ( ! empty( $boost_connection_message ) ) : ?>
                <div class="fsb-info-box">
                    <p><?php echo esc_html( $boost_connection_message ); ?></p>
                </div>
            <?php endif; ?>
        </div>

        <div class="fsb-card">
            <h2><?php echo esc_html( __( 'How Auto-Boost Works', '5th-social-bot' ) ); ?></h2>
        <ol>
            <li><?php echo esc_html__( 'You publish a post/page on your WordPress site.', '5th-social-bot' ); ?></li>
            <li>
                <?php
                printf(
                    /* translators: %s: link to social.5th.ro */
                    esc_html__( 'The plugin automatically posts it to your primary Mastodon account on %s.', '5th-social-bot' ),
                    '<a href="https://social.5th.ro" target="_blank" rel="noopener noreferrer">social.5th.ro</a>'
                );
                ?>
            </li>
                <li><?php echo esc_html__( 'If auto-boost is enabled, the plugin automatically boosts (reblogs) that post on your second Mastodon instance.', '5th-social-bot' ); ?></li>
                <li><?php echo esc_html__( 'The boost link is saved and displayed in the Bulk & queue tab next to the original post link.', '5th-social-bot' ); ?></li>
            </ol>
            
            <div class="fsb-info-box">
                <p style="margin:0;">
                    <strong><?php echo esc_html__( 'Important:', '5th-social-bot' ); ?></strong>
                    <?php
                    printf(
                        /* translators: %s: link to social.5th.ro */
                        esc_html__( 'The second instance only boosts posts that were successfully published on %s. It does not create new posts - it only reblogs existing posts from your primary account.', '5th-social-bot' ),
                        '<a href="https://social.5th.ro" target="_blank" rel="noopener noreferrer">social.5th.ro</a>'
                    );
                    ?>
                </p>
            </div>
        </div>

        <p style="margin-top: 2em; font-size: 12px; color: #666;">
            <?php
            printf(
                /* translators: %1$s: link to social.5th.ro, %2$s: link to plugin.5th.ro */
                wp_kses_post( __( 'Plugin created for WordPress → Mastodon (%1$s) integration. Official website: %2$s', '5th-social-bot' ) ),
                '<a href="https://social.5th.ro" target="_blank" rel="noopener noreferrer">social.5th.ro</a>',
                '<a href="https://plugin.5th.ro" target="_blank" rel="noopener noreferrer">plugin.5th.ro</a>'
            );
            ?>
            <br />
            <?php
            echo wp_kses_post(
                'Signed: <strong>SC ARC MEDIASOFT SRL</strong> — <a href="https://e-agentie.ro" target="_blank" rel="noopener noreferrer">e-agentie.ro</a>'
            );
            ?>
        </p>

        <script>
        (function(){
            // Toggle boost token visibility.
            var boostTokenInput = document.getElementById('fsb_boost_access_token');
            var boostToggleBtn = document.getElementById('fsb-toggle-boost-token');
            var boostToggleText = document.getElementById('fsb-toggle-boost-token-text');
            
            if (boostTokenInput && boostToggleBtn && boostToggleText) {
                boostToggleBtn.addEventListener('click', function(e){
                    e.preventDefault();
                    if (boostTokenInput.type === 'password') {
                        boostTokenInput.type = 'text';
                        boostToggleText.textContent = '<?php echo esc_js( __( 'Hide', '5th-social-bot' ) ); ?>';
                    } else {
                        boostTokenInput.type = 'password';
                        boostToggleText.textContent = '<?php echo esc_js( __( 'Show', '5th-social-bot' ) ); ?>';
                    }
                });
            }
        })();
        </script>
        <?php
    }

    /**
     * Import stats from Mastodon account.
     *
     * @param array  $settings Settings array.
     * @param string $instance_url Instance URL.
     * @return array Result array with success, count, and message.
     */
    protected static function import_stats_from_mastodon( $settings, $instance_url ) {
        $result = array(
            'success' => false,
            'count'   => 0,
            'message' => '',
        );

        if ( empty( $settings['access_token'] ) ) {
            $result['message'] = __( 'Access token is not set.', '5th-social-bot' );
            return $result;
        }

        $access_token = trim( $settings['access_token'] );
        $api_url = trailingslashit( $instance_url ) . 'api/v1/accounts/verify_credentials';

        // Get account ID first.
        $response = wp_remote_get(
            $api_url,
            array(
                'timeout' => 15,
                'headers' => array(
                    'Authorization' => 'Bearer ' . $access_token,
                ),
            )
        );

        if ( is_wp_error( $response ) ) {
            $result['message'] = __( 'Failed to connect to Mastodon: ', '5th-social-bot' ) . $response->get_error_message();
            return $result;
        }

        $code = wp_remote_retrieve_response_code( $response );
        if ( $code < 200 || $code >= 300 ) {
            $result['message'] = __( 'Failed to verify credentials. HTTP ', '5th-social-bot' ) . $code;
            return $result;
        }

        $body = wp_remote_retrieve_body( $response );
        $account_data = json_decode( $body, true );
        if ( ! is_array( $account_data ) || empty( $account_data['id'] ) ) {
            $result['message'] = __( 'Invalid response from Mastodon API.', '5th-social-bot' );
            return $result;
        }

        $account_id = $account_data['id'];
        $statuses_url = trailingslashit( $instance_url ) . 'api/v1/accounts/' . absint( $account_id ) . '/statuses?limit=40';

        // Fetch statuses.
        $response = wp_remote_get(
            $statuses_url,
            array(
                'timeout' => 15,
                'headers' => array(
                    'Authorization' => 'Bearer ' . $access_token,
                ),
            )
        );

        if ( is_wp_error( $response ) ) {
            $result['message'] = __( 'Failed to fetch statuses: ', '5th-social-bot' ) . $response->get_error_message();
            return $result;
        }

        $code = wp_remote_retrieve_response_code( $response );
        if ( $code < 200 || $code >= 300 ) {
            $result['message'] = __( 'Failed to fetch statuses. HTTP ', '5th-social-bot' ) . $code;
            return $result;
        }

        $body = wp_remote_retrieve_body( $response );
        $statuses = json_decode( $body, true );
        if ( ! is_array( $statuses ) ) {
            $result['message'] = __( 'Invalid response from Mastodon API.', '5th-social-bot' );
            return $result;
        }

        $imported = 0;
        $site_domain = wp_parse_url( home_url(), PHP_URL_HOST );
        
        // Get all published posts that don't have a Mastodon URL yet.
        $posts = get_posts(
            array(
                'post_type'      => array( 'post', 'page' ),
                'post_status'    => 'publish',
                'posts_per_page' => -1,
                // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query -- Necessary for finding posts without Mastodon URLs.
                'meta_query'     => array(
                    'relation' => 'OR',
                    array(
                        'key'     => '_fifth_social_bot_mastodon_url',
                        'value'   => '',
                        'compare' => '=',
                    ),
                    array(
                        'key'     => '_fifth_social_bot_mastodon_url',
                        'compare' => 'NOT EXISTS',
                    ),
                ),
            )
        );

        foreach ( $statuses as $status ) {
            if ( empty( $status['url'] ) ) {
                continue;
            }

            $mastodon_url = sanitize_url( $status['url'] );
            $status_content = isset( $status['content'] ) ? wp_strip_all_tags( $status['content'] ) : '';
            
            // Try to match by checking if status content contains post URL.
            foreach ( $posts as $post ) {
                $permalink = get_permalink( $post->ID );
                $permalink_clean = str_replace( array( 'https://', 'http://' ), '', $permalink );
                
                // Check if status content contains the post URL (with or without protocol).
                if ( ! empty( $permalink_clean ) && ( strpos( $status_content, $permalink ) !== false || strpos( $status_content, $permalink_clean ) !== false ) ) {
                    // Found a match - update the Mastodon URL.
                    update_post_meta( $post->ID, '_fifth_social_bot_mastodon_url', $mastodon_url );
                    update_post_meta( $post->ID, '_fifth_social_bot_posted', '1' );
                    $imported++;
                    
                    // Remove from array to avoid duplicate matches.
                    $key = array_search( $post, $posts, true );
                    if ( false !== $key ) {
                        unset( $posts[ $key ] );
                    }
                    break;
                }
            }
        }

        $result['success'] = true;
        $result['count'] = $imported;
        $result['message'] = sprintf(
            /* translators: %d: number of posts */
            __( 'Imported %d post URLs from Mastodon.', '5th-social-bot' ),
            $imported
        );

        return $result;
    }
}
