<?php
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Core logic: settings, composing statuses and sending to Mastodon.
 */
class Fifth_Social_Bot_Core {

    public function __construct() {
        add_action( 'publish_post', array( $this, 'handle_publish' ), 10, 2 );
        add_action( 'publish_page', array( $this, 'handle_publish' ), 10, 2 );
    }

    /**
     * Helper: UTF-8 safe strlen.
     *
     * @param string $string Text.
     * @return int
     */
    protected static function str_length( $string ) {
        if ( function_exists( 'mb_strlen' ) ) {
            return mb_strlen( $string, 'UTF-8' );
        }
        return strlen( $string );
    }

    /**
     * Helper: UTF-8 safe substr.
     *
     * @param string   $string Text.
     * @param int      $start  Start.
     * @param int|null $length Length or null.
     * @return string
     */
    protected static function str_substr( $string, $start, $length = null ) {
        if ( function_exists( 'mb_substr' ) ) {
            if ( null === $length ) {
                return mb_substr( $string, $start, null, 'UTF-8' );
            }
            return mb_substr( $string, $start, $length, 'UTF-8' );
        }
        if ( null === $length ) {
            return substr( $string, $start );
        }
        return substr( $string, $start, $length );
    }

    /**
     * Default template.
     *
     * @return string
     */
    public static function get_default_template() {
        return "{text}\n{tags}\n{url}";
    }

    /**
     * Default "read more" texts.
     *
     * @return array
     */
    public static function get_default_readmore_texts() {
        return array(
            __( 'Read more at:', '5th-social-bot' ),
            __( 'More details at:', '5th-social-bot' ),
        );
    }

    /**
     * Get plugin settings.
     *
     * @return array
     */
    public static function get_settings() {
        $defaults = array(
            'access_token'      => '',
            'visibility'        => 'public',
            'language'          => 'ro',
            'post_posts'        => 1,
            'post_pages'        => 1,
            'template'          => self::get_default_template(),
            'readmore_texts'    => implode( "\n", self::get_default_readmore_texts() ),
            'mastodon_username' => '',
            'enable_verification' => 0,
            'debug_mode' => 0,
            'boost_enabled' => 0,
            'boost_instance' => '',
            'boost_access_token' => '',
            'boost_username' => '',
            'boost_max_per_day' => 10,
            'boost_min_interval' => 30, // minutes
            'use_custom_instance' => 0,
            'mastodon_instance' => '',
        );

        $settings = get_option( 'fifth_social_bot_settings', array() );
        if ( ! is_array( $settings ) ) {
            $settings = array();
        }

        $settings = array_merge( $defaults, $settings );

        if ( ! in_array( $settings['visibility'], array( 'public', 'unlisted', 'private', 'direct' ), true ) ) {
            $settings['visibility'] = 'public';
        }

        return $settings;
    }

    /**
     * Get Mastodon instance URL from settings or default.
     *
     * @param array|null $settings Settings array. If null, will fetch from options.
     * @return string Instance URL.
     */
    public static function get_instance_url( $settings = null ) {
        if ( null === $settings ) {
            $settings = self::get_settings();
        }

        // If custom instance is enabled and URL is provided, use it.
        if ( ! empty( $settings['use_custom_instance'] ) && ! empty( $settings['mastodon_instance'] ) ) {
            $instance = trim( $settings['mastodon_instance'] );
            // Ensure it starts with http:// or https://
            if ( ! preg_match( '#^https?://#', $instance ) ) {
                $instance = 'https://' . $instance;
            }
            // Remove trailing slash
            $instance = rtrim( $instance, '/' );
            return $instance;
        }

        // Default instance
        return FIFTH_SOCIAL_BOT_INSTANCE;
    }

    /**
     * Save settings.
     *
     * @param array $settings Settings.
     */
    public static function save_settings( $settings ) {
        if ( ! is_array( $settings ) ) {
            $settings = array();
        }
        update_option( 'fifth_social_bot_settings', $settings );
    }

    /**
     * Normalize template. Convert \n or \r\n to actual newlines.
     *
     * @param array|null $settings Settings.
     * @return string
     */
    public static function get_template( $settings = null ) {
        if ( null === $settings ) {
            $settings = self::get_settings();
        }

        $template = '';
        if ( isset( $settings['template'] ) ) {
            $template = (string) $settings['template'];
        }

        $template = trim( $template );
        if ( '' === $template ) {
            $template = self::get_default_template();
        }

        // Convert literal \n sequences (backslash + n) to real newlines.
        $template = strtr(
            $template,
            array(
                '\\r\\n' => "\n",
                '\\n'    => "\n",
            )
        );

       return $template;
    }

    /**
     * Pick one read-more text (random).
     *
     * @param array|null $settings Settings.
     * @return string
     */
    public static function get_readmore_text( $settings = null ) {
        if ( null === $settings ) {
            $settings = self::get_settings();
        }

        $raw = '';
        if ( isset( $settings['readmore_texts'] ) ) {
            $raw = (string) $settings['readmore_texts'];
        }

        $items = array();

        if ( '' !== $raw ) {
            $lines = preg_split( '/\r\n|\r|\n/', $raw );
            if ( is_array( $lines ) ) {
                foreach ( $lines as $line ) {
                    $line = trim( $line );
                    if ( '' === $line ) {
                        continue;
                    }
                    $line = strtr(
                        $line,
                        array(
                            '\\r\\n' => "\n",
                            '\\n'    => "\n",
                        )
                    );
                    $items[] = $line;
                }
            }
        }

        if ( empty( $items ) ) {
            $items = self::get_default_readmore_texts();
        }

        return $items[ array_rand( $items ) ];
    }

    /**
     * Build hashtags from WP tags.
     *
     * @param int $post_id Post ID.
     * @return array
     */
    public static function build_hashtags( $post_id ) {
        $tags = wp_get_post_tags( $post_id );
        if ( empty( $tags ) ) {
            return array();
        }

        $hashtags = array();

        foreach ( $tags as $tag ) {
            $name = $tag->name;
            $name = html_entity_decode( $name, ENT_QUOTES, 'UTF-8' );

            // Remove spaces, -, _, ;, :
            $name = preg_replace( '/[\s\-\_\;\:]+/u', '', $name );

            // Remove any other non-letter/number.
            $name = preg_replace( '/[^0-9\p{L}]+/u', '', $name );

            $name = trim( $name );
            if ( '' === $name ) {
                continue;
            }

            $hashtags[] = '#' . $name;
        }

        $hashtags = array_unique( $hashtags );
        return $hashtags;
    }

    /**
     * Apply template with placeholders.
     *
     * @param string $template      Template.
     * @param string $text          Text.
     * @param string $tags          Tags string.
     * @param string $readmore_text Read-more text.
     * @param string $url           URL.
     *
     * @return string
     */
    protected static function apply_template( $template, $text, $tags, $readmore_text, $url ) {
        $replacements = array(
            '{text}'          => $text,
            '{tags}'          => $tags,
            '{readmore_text}' => $readmore_text,
            '{url}'           => $url,
        );

        $status = strtr( $template, $replacements );

        // Collapse multiple spaces/tabs to a single space (but keep newlines).
        $status = preg_replace( '/[ \t]+/', ' ', $status );

        // Trim each line.
        $lines = preg_split( '/\r\n|\r|\n/', $status );
        if ( is_array( $lines ) ) {
            $lines = array_map( 'trim', $lines );
            $status = implode( "\n", $lines );
        }

        return trim( $status );
    }

    /**
     * Compose Mastodon status (max 500 chars).
     *
     * @param int        $post_id  Post ID.
     * @param array|null $settings Settings.
     * @return string
     */
    public static function compose_status( $post_id, $settings = null ) {
        if ( null === $settings ) {
            $settings = self::get_settings();
        }

        $post = get_post( $post_id );
        if ( ! $post ) {
            return '';
        }

        if ( has_excerpt( $post_id ) ) {
            $excerpt = get_the_excerpt( $post_id );
        } else {
            $excerpt = wp_trim_words( wp_strip_all_tags( $post->post_content ), 55 );
        }

        $excerpt = html_entity_decode( $excerpt, ENT_QUOTES, 'UTF-8' );
        $excerpt = preg_replace( '/\s+/', ' ', $excerpt );
        $excerpt = trim( $excerpt );

        $permalink = get_permalink( $post_id );
        $hashtags  = self::build_hashtags( $post_id );
        $tags_str  = empty( $hashtags ) ? '' : implode( ' ', $hashtags );

        $readmore_text = self::get_readmore_text( $settings );
        $template      = self::get_template( $settings );

        $status     = self::apply_template( $template, $excerpt, $tags_str, $readmore_text, $permalink );
        $max_length = 500;

        if ( self::str_length( $status ) <= $max_length ) {
            return $status;
        }

        // Try shortening only {text}.
        $status_no_text = self::apply_template( $template, '', $tags_str, $readmore_text, $permalink );
        $base_len       = self::str_length( $status_no_text );
        $max_text_len   = $max_length - $base_len;

        if ( $max_text_len <= 0 ) {
            $fallback = '.... ' . $readmore_text . ' ' . $permalink;
            if ( self::str_length( $fallback ) > $max_length ) {
                $extra   = self::str_length( $fallback ) - $max_length;
                $new_len = max( 0, self::str_length( $readmore_text ) - $extra - 5 );
                if ( $new_len > 0 ) {
                    $readmore_text = self::str_substr( $readmore_text, 0, $new_len );
                    $readmore_text = preg_replace( '/\s+\S*$/u', '', $readmore_text );
                    $readmore_text = trim( $readmore_text );
                } else {
                    $readmore_text = '';
                }
                $fallback = '.... ' . $readmore_text . ' ' . $permalink;
            }
            return trim( $fallback );
        }

        if ( self::str_length( $excerpt ) > $max_text_len ) {
            $excerpt = self::str_substr( $excerpt, 0, $max_text_len );
            $excerpt = preg_replace( '/\s+\S*$/u', '', $excerpt );
            $excerpt = trim( $excerpt );
        }

        $status = self::apply_template( $template, $excerpt, $tags_str, $readmore_text, $permalink );

        if ( self::str_length( $status ) > $max_length ) {
            $status = $status_no_text;
            if ( self::str_length( $status ) > $max_length ) {
                $fallback = '.... ' . $readmore_text . ' ' . $permalink;
                if ( self::str_length( $fallback ) <= $max_length ) {
                    return $fallback;
                }
                return self::str_substr( $fallback, 0, $max_length );
            }
        }

        return $status;
    }

    /**
     * Send status to Mastodon.
     *
     * @param int        $post_id  Post ID.
     * @param array|null $settings Settings.
     * @return bool
     */
    /**
     * Check if debug mode is enabled.
     *
     * @return bool
     */
    private static function is_debug_mode() {
        $settings = self::get_settings();
        return ! empty( $settings['debug_mode'] );
    }
    
    /**
     * Log debug message if debug mode is enabled.
     *
     * @param string $message Message to log.
     * @param string $level   Log level (info|error).
     */
    private static function debug_log( $message, $level = 'info' ) {
        if ( self::is_debug_mode() ) {
            Fifth_Social_Bot_Logger::log( '[Debug] ' . $message, $level );
        }
    }

    public static function send_status( $post_id, $settings = null ) {
        if ( null === $settings ) {
            $settings = self::get_settings();
        }
        
        self::debug_log( 'send_status called for post ID: ' . $post_id );

        if ( empty( $settings['access_token'] ) ) {
            Fifth_Social_Bot_Logger::log( 'Missing access token, cannot send status for post ID ' . $post_id, 'error' );
            return false;
        }

        $post = get_post( $post_id );
        if ( ! $post || 'publish' !== $post->post_status ) {
            Fifth_Social_Bot_Logger::log( 'Post not published or not found (ID ' . $post_id . ').', 'error' );
            return false;
        }

        if ( '1' === get_post_meta( $post_id, '_fifth_social_bot_posted', true ) ) {
            Fifth_Social_Bot_Logger::log( 'Post already marked as posted (ID ' . $post_id . '). Skipping.', 'info' );
            return false;
        }

        $status = self::compose_status( $post_id, $settings );
        if ( '' === $status ) {
            Fifth_Social_Bot_Logger::log( 'Empty status for post ID ' . $post_id, 'error' );
            return false;
        }
        
        self::debug_log( 'Composed status for post ID ' . $post_id . ': ' . substr( $status, 0, 100 ) . '...' );

        $visibility = isset( $settings['visibility'] ) ? $settings['visibility'] : 'public';
        if ( ! in_array( $visibility, array( 'public', 'unlisted', 'private', 'direct' ), true ) ) {
            $visibility = 'public';
        }
        
        self::debug_log( 'Sending status to Mastodon API with visibility: ' . $visibility );

        $body = array(
            'status'     => $status,
            'visibility' => $visibility,
        );

        // Add language parameter if set.
        $language = isset( $settings['language'] ) ? trim( $settings['language'] ) : 'ro';
        if ( ! empty( $language ) && preg_match( '/^[a-z]{2,3}$/i', $language ) ) {
            $body['language'] = strtolower( $language );
            self::debug_log( 'Setting language to: ' . $body['language'] );
        }

        $instance_url = self::get_instance_url( $settings );
        $api_url = trailingslashit( $instance_url ) . 'api/v1/statuses';
        self::debug_log( 'API URL: ' . $api_url );
        
        // Clean and validate access token.
        $access_token = trim( $settings['access_token'] );
        if ( empty( $access_token ) ) {
            Fifth_Social_Bot_Logger::log( 'Access token is empty for post ID ' . $post_id, 'error' );
            return false;
        }
        
        // Remove any extra whitespace or newlines from token.
        $access_token = preg_replace( '/\s+/', '', $access_token );
        
        self::debug_log( 'Access token length: ' . strlen( $access_token ) . ' characters' );
        self::debug_log( 'Access token preview: ' . substr( $access_token, 0, 10 ) . '...' . substr( $access_token, -5 ) );
        
        $response = wp_remote_post(
            $api_url,
            array(
                'timeout' => 15,
                'headers' => array(
                    'Authorization' => 'Bearer ' . $access_token,
                    'Content-Type'  => 'application/x-www-form-urlencoded',
                ),
                'body'    => $body,
            )
        );

        if ( is_wp_error( $response ) ) {
            self::debug_log( 'API request failed with WP_Error: ' . $response->get_error_message(), 'error' );
            Fifth_Social_Bot_Logger::log(
                'WP_Error while posting to Mastodon for post ID ' . $post_id . ': ' . $response->get_error_message(),
                'error'
            );
            return false;
        }

        $code = wp_remote_retrieve_response_code( $response );
        self::debug_log( 'API response code: ' . $code );
        
        if ( $code < 200 || $code >= 300 ) {
            $response_body = wp_remote_retrieve_body( $response );
            self::debug_log( 'API error response body: ' . substr( $response_body, 0, 200 ), 'error' );
            
            // Special handling for 401 (Unauthorized) - token invalid or expired.
            if ( 401 === $code ) {
                $error_message = __( 'Access token is invalid or expired. Please generate a new token in your Mastodon account and update it in Settings.', '5th-social-bot' );
                Fifth_Social_Bot_Logger::log(
                    'HTTP 401 (Unauthorized) from Mastodon for post ID ' . $post_id . '. ' . $error_message . ' Response: ' . $response_body,
                    'error'
                );
            } else {
                Fifth_Social_Bot_Logger::log(
                    'HTTP ' . $code . ' from Mastodon for post ID ' . $post_id . '. Body: ' . $response_body,
                    'error'
                );
            }
            return false;
        }

        // Extract Mastodon post URL from response.
        $body = wp_remote_retrieve_body( $response );
        $data = json_decode( $body, true );
        $mastodon_url = '';
        if ( is_array( $data ) && isset( $data['url'] ) ) {
            $mastodon_url = sanitize_url( $data['url'] );
            if ( ! empty( $mastodon_url ) ) {
                update_post_meta( $post_id, '_fifth_social_bot_mastodon_url', $mastodon_url );
            }
        }

        update_post_meta( $post_id, '_fifth_social_bot_posted', '1' );
        delete_post_meta( $post_id, '_fifth_social_bot_queued' );

        $log_message = 'Post sent to Mastodon successfully (ID ' . $post_id . ').';
        if ( ! empty( $mastodon_url ) ) {
            $log_message .= ' URL: ' . $mastodon_url;
        }
        Fifth_Social_Bot_Logger::log( $log_message, 'info' );

        // Auto-boost on second instance if enabled and URL is available.
        if ( ! empty( $mastodon_url ) ) {
            // Try to boost immediately, but if rate limited, add to queue.
            $boost_result = Fifth_Social_Bot_Boost::boost_post( $mastodon_url, $post_id );
            if ( $boost_result['success'] && ! empty( $boost_result['url'] ) ) {
                update_post_meta( $post_id, '_fifth_social_bot_boost_url', $boost_result['url'] );
                delete_post_meta( $post_id, '_fifth_social_bot_boost_queued' );
                delete_post_meta( $post_id, '_fifth_social_bot_boost_next_time' );
                self::debug_log( 'Auto-boost successful for post ID ' . $post_id . '. Boost URL: ' . $boost_result['url'] );
            } else {
                // Rate limited or failed - add to boost queue.
                Fifth_Social_Bot_Boost::enqueue_boost( $post_id );
                if ( ! empty( $boost_result['message'] ) ) {
                    self::debug_log( 'Auto-boost queued for post ID ' . $post_id . ': ' . $boost_result['message'], 'info' );
                }
            }
        }

        return true;
    }

    /**
     * Send a test status to Mastodon (for testing purposes).
     *
     * @param string $test_status Test status message.
     * @param array  $settings    Settings array.
     * @return array { success: bool, url: string, message: string }
     */
    public static function send_test_status( $test_status, $settings = null ) {
        if ( null === $settings ) {
            $settings = self::get_settings();
        }

        self::debug_log( 'send_test_status called. Test status length: ' . strlen( $test_status ) );

        $result = array(
            'success' => false,
            'url'     => '',
            'message' => '',
        );

        if ( empty( $settings['access_token'] ) ) {
            $result['message'] = __( 'Access token is not set.', '5th-social-bot' );
            Fifth_Social_Bot_Logger::log( 'Test post failed: Access token is not set.', 'error' );
            return $result;
        }

        $visibility = isset( $settings['visibility'] ) ? $settings['visibility'] : 'public';
        if ( ! in_array( $visibility, array( 'public', 'unlisted', 'private', 'direct' ), true ) ) {
            $visibility = 'public';
        }

        self::debug_log( 'Sending test status to Mastodon API with visibility: ' . $visibility );

        $body = array(
            'status'     => $test_status,
            'visibility' => $visibility,
        );

        // Add language parameter if set.
        $language = isset( $settings['language'] ) ? trim( $settings['language'] ) : 'ro';
        if ( ! empty( $language ) && preg_match( '/^[a-z]{2,3}$/i', $language ) ) {
            $body['language'] = strtolower( $language );
            self::debug_log( 'Setting language to: ' . $body['language'] );
        }

        $instance_url = self::get_instance_url( $settings );
        $api_url = trailingslashit( $instance_url ) . 'api/v1/statuses';
        self::debug_log( 'API URL: ' . $api_url );
        
        // Clean and validate access token.
        $access_token = trim( $settings['access_token'] );
        $access_token = preg_replace( '/\s+/', '', $access_token );

        self::debug_log( 'Access token length: ' . strlen( $access_token ) . ' characters' );
        self::debug_log( 'Access token preview: ' . substr( $access_token, 0, 10 ) . '...' . substr( $access_token, -5 ) );

        $response = wp_remote_post(
            $api_url,
            array(
                'timeout' => 15,
                'headers' => array(
                    'Authorization' => 'Bearer ' . $access_token,
                    'Content-Type'  => 'application/x-www-form-urlencoded',
                ),
                'body'    => $body,
            )
        );

        if ( is_wp_error( $response ) ) {
            $error_message = $response->get_error_message();
            $result['message'] = $error_message;
            self::debug_log( 'Test post API request failed with WP_Error: ' . $error_message, 'error' );
            Fifth_Social_Bot_Logger::log( 'Test post failed: WP_Error - ' . $error_message, 'error' );
            return $result;
        }

        $code = wp_remote_retrieve_response_code( $response );
        self::debug_log( 'Test post API response code: ' . $code );
        
        if ( $code < 200 || $code >= 300 ) {
            $response_body = wp_remote_retrieve_body( $response );
            self::debug_log( 'Test post API error response body: ' . substr( $response_body, 0, 200 ), 'error' );
            
            $error_data = json_decode( $response_body, true );
            if ( is_array( $error_data ) && isset( $error_data['error'] ) ) {
                $result['message'] = $error_data['error'];
            } else {
                $result['message'] = sprintf(
                    /* translators: %d: HTTP status code */
                    __( 'HTTP %d error from Mastodon API.', '5th-social-bot' ),
                    $code
                );
            }
            
            // Special handling for 401 (Unauthorized) - token invalid or expired.
            if ( 401 === $code ) {
                $error_message = __( 'Access token is invalid or expired. Please generate a new token in your Mastodon account and update it in Settings.', '5th-social-bot' );
                Fifth_Social_Bot_Logger::log( 'Test post failed: HTTP 401 (Unauthorized). ' . $error_message . ' Response: ' . $response_body, 'error' );
            } else {
                Fifth_Social_Bot_Logger::log( 'Test post failed: HTTP ' . $code . '. Response: ' . $response_body, 'error' );
            }
            
            return $result;
        }

        // Extract Mastodon post URL from response.
        $response_body = wp_remote_retrieve_body( $response );
        $data = json_decode( $response_body, true );
        if ( is_array( $data ) && isset( $data['url'] ) ) {
            $result['success'] = true;
            $result['url']     = sanitize_url( $data['url'] );
            $result['message'] = __( 'Test post sent successfully.', '5th-social-bot' );
            self::debug_log( 'Test post sent successfully. URL: ' . $result['url'] );
            Fifth_Social_Bot_Logger::log( 'Test post sent successfully: ' . $result['url'], 'info' );
        } else {
            $result['message'] = __( 'Test post sent, but could not retrieve post URL from response.', '5th-social-bot' );
            self::debug_log( 'Test post sent, but could not retrieve post URL from response. Response body: ' . substr( $response_body, 0, 200 ), 'error' );
            Fifth_Social_Bot_Logger::log( 'Test post sent, but could not retrieve post URL from response. Response: ' . $response_body, 'error' );
        }

        return $result;
    }

    /**
     * Test connection to Mastodon using the given token.
     *
     * @param string $access_token Access token.
     * @param string|null $instance_url Optional instance URL. If not provided, uses settings.
     * @return array { ok: bool, message: string, username: string, instance: string }
     */
    public static function test_connection( $access_token, $instance_url = null ) {
        $result = array(
            'ok'       => false,
            'message'  => '',
            'username' => '',
            'instance' => '',
        );

        if ( empty( $access_token ) ) {
            $result['message'] = __( 'Missing access token.', '5th-social-bot' );
            return $result;
        }

        if ( null === $instance_url ) {
            $instance_url = self::get_instance_url();
        }
        $result['instance'] = $instance_url;
        $url = trailingslashit( $instance_url ) . 'api/v1/accounts/verify_credentials';

        $response = wp_remote_get(
            $url,
            array(
                'timeout' => 15,
                'headers' => array(
                    'Authorization' => 'Bearer ' . $access_token,
                ),
            )
        );

        if ( is_wp_error( $response ) ) {
            $result['message'] = 'WP error: ' . $response->get_error_message();
            return $result;
        }

        $code = wp_remote_retrieve_response_code( $response );
        if ( $code < 200 || $code >= 300 ) {
            $result['message'] = 'HTTP ' . $code . '. Response: ' . wp_remote_retrieve_body( $response );
            return $result;
        }

        $body = wp_remote_retrieve_body( $response );
        $data = json_decode( $body, true );
        if ( ! is_array( $data ) ) {
            $result['message'] = __( 'Invalid JSON response from Mastodon.', '5th-social-bot' );
            return $result;
        }

        $username = isset( $data['username'] ) ? trim( $data['username'] ) : '';
        $acct     = isset( $data['acct'] ) ? trim( $data['acct'] ) : '';

        // Extract username from acct if it contains @instance format.
        if ( ! empty( $acct ) && strpos( $acct, '@' ) !== false ) {
            $parts = explode( '@', $acct );
            if ( ! empty( $parts[0] ) ) {
                $username = $parts[0];
            }
        }

        $result['username'] = $username;
        $result['message']  = __( 'Connection OK.', '5th-social-bot' ) . ' User: ' . $username . ' (' . $acct . ').';
        $result['ok']       = true;

        return $result;
    }

    /**
     * Handle post/page publish.
     *
     * @param int     $post_id Post ID.
     * @param WP_Post $post    Post object.
     */
    public function handle_publish( $post_id, $post ) {
        if ( wp_is_post_autosave( $post_id ) || wp_is_post_revision( $post_id ) ) {
            return;
        }

        // Check per-post control: if excluded, don't post.
        $exclude = get_post_meta( $post_id, '_fifth_social_bot_exclude', true );
        if ( ! empty( $exclude ) ) {
            Fifth_Social_Bot_Logger::log(
                'Post ID ' . $post_id . ' excluded from Mastodon auto-posting (per-post control).',
                'info'
            );
            return;
        }

        $settings = self::get_settings();

        if ( 'post' === $post->post_type && empty( $settings['post_posts'] ) ) {
            return;
        }

        if ( 'page' === $post->post_type && empty( $settings['post_pages'] ) ) {
            return;
        }

        if ( '1' === get_post_meta( $post_id, '_fifth_social_bot_posted', true ) ) {
            return;
        }

        self::send_status( $post_id, $settings );
    }
}